#include "ages.h"

#include <util/datetime/parser.h>
#include <util/string/printf.h>
#include <util/string/cast.h>
#include <util/generic/algorithm.h>
#include <util/digest/multi.h>
#include <util/digest/numeric.h>

namespace NTravel {

static const int g_MaxChildAge = 21;
static const int g_SpecialChildAge = 255;
static const int g_MaxStringLengthToParse = 1000;
static const int g_MaxAdults = 100;
static const int g_MaxChildren = 100;

TAges TAges::Empty() {
    return TAges();
}

TAges TAges::FromAgesString(TStringBuf ages, bool allowTrailers) {
    if (!ages) {
        throw TInvalidAgesException() << "Empty ages";
    }
    if (ages.length() > g_MaxStringLengthToParse) { // avoid parsing too long strings
        throw TInvalidAgesException() << "Too long ages string";
    }
    size_t adultCount = 0;
    TVector<size_t> childrenAges;
    while (!ages.empty()) {
        TStringBuf ageBuf = ages.NextTok(',');
        if (allowTrailers) {
            ageBuf = ageBuf.NextTok(".");
        }
        int age;
        try {
            age = ::FromString(ageBuf);
        } catch (const TFromStringException&) {
            throw TInvalidAgesException() << CurrentExceptionMessage();
        }
        if (age == 88) {
            ++adultCount;
        } else if (age >= 0 && age <= g_MaxChildAge) {
            childrenAges.push_back(age);
        } else {
            throw TInvalidAgesException() << "Invalid age " << (int)age;
        }
    }
    return TAges(adultCount, childrenAges);
}

TString TAges::ToAgesString() const {
    TString res;
    for (auto ag: ChildrenAges) {
        if (res) {
            res += ",";
        }
        res += ::ToString(ag);
    }
    for (size_t i = 0; i < AdultCount; ++i) {
        if (res) {
            res += ",";
        }
        res += "88";
    }
    return res;
}

TAges TAges::FromOccupancyString(TStringBuf occupancy) {
    if (!occupancy) {
        throw TInvalidAgesException() << "Empty occupancy";
    }
    if (occupancy.length() > g_MaxStringLengthToParse) { // avoid parsing too long strings
        throw TInvalidAgesException() << "Too long occupancy string";
    }
    size_t adultCount = 0;
    TVector<size_t> childrenAges;
    TStringBuf adultsStr = occupancy.NextTok('-');
    try {
        adultCount = ::FromString(adultsStr);
    } catch (const TFromStringException&) {
        throw TInvalidAgesException() << CurrentExceptionMessage();
    }
    while (!occupancy.empty()) {
        TStringBuf ageBuf = occupancy.NextTok(',');
        int age;
        try {
            age = ::FromString(ageBuf);
        } catch (const TFromStringException&) {
            throw TInvalidAgesException() << CurrentExceptionMessage();
        }
        if (age == g_SpecialChildAge) {
            age = g_MaxChildAge;
        }
        if (age >= 0 && age <= g_MaxChildAge) {
            childrenAges.push_back(age);
        } else {
            throw TInvalidAgesException() << "Invalid age " << (int)age;
        }
    }
    TAges res(adultCount, childrenAges);
    if (res.IsEmpty()) {
        throw TInvalidAgesException() << "Empty occupancy";
    }
    return res;
}

TString TAges::ToOccupancyString() const {
    TString res = ::ToString(AdultCount);
    if (ChildrenAges) {
        res += "-";
        bool first = true;
        for (auto ag: ChildrenAges) {
            if (first) {
                first = false;
            } else {
                res += ",";
            }
            res += ::ToString(ag);
        }
    }
    return res;
}

TAges TAges::FromAdultCount(size_t adultCount) {
    return TAges(adultCount, {});
}

bool TAges::IsEmpty() const {
    return AdultCount == 0 && ChildrenAges.empty();
}

size_t TAges::GetAdultCount() const {
    return AdultCount;
}

bool TAges::HasChildren() const {
    return !ChildrenAges.empty();
}

const TVector<size_t>& TAges::GetChildrenAges() const {
    return ChildrenAges;
}

size_t TAges::GetChildAgeMin() const {
   if (HasChildren()) {
       return *MinElement(ChildrenAges.begin(), ChildrenAges.end());
   }
   return 0;
}

size_t TAges::GetChildAgeMax() const {
    if (HasChildren()) {
        return *MaxElement(ChildrenAges.begin(), ChildrenAges.end());
    }
    return 0;
}

bool TAges::operator ==(const TAges& rhs) const {
    return AdultCount == rhs.AdultCount && ChildrenAges == rhs.ChildrenAges;
}

bool TAges::operator !=(const TAges& rhs) const {
    return !operator ==(rhs);
}

size_t TAges::Hash() const {
    size_t res = IntHash(AdultCount);
    for (auto ag: ChildrenAges) {
        res = CombineHashes(res, ag);
    }
    return res;
}

size_t TAges::GetAllocSize() const {
    return ChildrenAges.capacity() * sizeof ChildrenAges.back();
}

TAges::TAges(size_t adultCount, TVector<size_t> childrenAges)
    : AdultCount(adultCount)
    , ChildrenAges(std::move(childrenAges)) {
    if (AdultCount > g_MaxAdults) {
        throw TInvalidAgesException() << "Too many adults: " << AdultCount;
    }
    if (ChildrenAges.size() > g_MaxChildren) {
        throw TInvalidAgesException() << "Too many children: " << ChildrenAges.size();
    }
    Sort(std::begin(ChildrenAges), std::end(ChildrenAges)); // Для одинакового результата в случае "1,2" и "2,1"
}

//----------------TCapacity------------------------------

TCapacity TCapacity::FromCapacityString(TStringBuf capacity) {
    TCapacity res;
    TStringBuf pfx = capacity.SubStr(0, 2);
    if (pfx == "=="sv) {
        res.IsExactMatch = true;
    } else if (pfx == "<="sv) {
        res.IsExactMatch = false;
    } else {
        throw TInvalidAgesException() << "Inavlid capacity prefix: '" << pfx << "'";
    }
    res.Ages = TAges::FromOccupancyString(capacity.Skip(2));
    return res;
}

TString TCapacity::ToCapacityString() const {
    TString res;
    if (IsExactMatch) {
        res += "==";
    } else {
        res += "<=";
    }
    return res + Ages.ToOccupancyString();
}

TAges TCapacity::DowncastToAges() const {
    // See HOTELS-2998
    if (IsExactMatch) {
        return Ages;
    } else {
        // See HOTELS-2998
        TAges res;
        res.AdultCount = Ages.AdultCount;
        return res;
    }
}

TCapacity TCapacity::FromAges(const TAges& ages) {
    TCapacity res;
    res.IsExactMatch = true;
    res.Ages = ages;
    return res;
}

bool TCapacity::operator ==(const TCapacity& rhs) const {
    return IsExactMatch == rhs.IsExactMatch && Ages == rhs.Ages;
}

bool TCapacity::IsExact() const {
    return IsExactMatch;
}

size_t TCapacity::GetAdultCount() const {
    return Ages.GetAdultCount();
}

bool TCapacity::HasChildren() const {
    return Ages.HasChildren();
}

bool TCapacity::Matches(const TAges& rhsAges) const {
    if (IsExactMatch) {
        return Ages == rhsAges;
    }
    if (rhsAges.AdultCount > Ages.AdultCount) {
        return false;
    }

    auto itC = Ages.ChildrenAges.begin();
    auto itA = rhsAges.ChildrenAges.begin();
    while (itC != Ages.ChildrenAges.end() && itA != rhsAges.ChildrenAges.end()) {
        if (*itA <= *itC) {
            // Ням, этот ребенок нам подходит
            ++itA;
        }
        ++itC;
    }
    size_t extraChildrenCount = std::distance(itA, rhsAges.ChildrenAges.end());
    if (rhsAges.AdultCount + extraChildrenCount > Ages.AdultCount) {
        return false;
    }
    return true;
}

bool TCapacity::Matches(size_t adultsFrom, size_t adultsTo, bool allowChildren) const {
    if (!allowChildren && HasChildren()) {
        return false;
    }
    if (adultsFrom > 0 && Ages.AdultCount < adultsFrom) {
        return false;
    }
    if (adultsTo > 0 && Ages.AdultCount > adultsTo) {
        return false;
    }
    // Exact не важен
    return true;
}

size_t TCapacity::Hash() const {
    return MultiHash(IsExactMatch, Ages);
}

size_t TCapacity::GetAllocSize() const {
    return Ages.GetAllocSize();
}

} // NTravel
