#pragma once

#include <util/generic/string.h>
#include <util/generic/yexception.h>
#include <util/generic/vector.h>

/* Библиотека для хранения и работы со всякими возрастами.
 *
 * Ages: Строка вида "2,3,88,88" - список возрастов. 88 - взрослый
 * Occupancy: Строка вида "A[-C1[,C2[,C3...]]]', где A - количество взрослых, Cx - возраста детей
 * Capacity = <f> + Occupancy, где <f> - функция сравнения, oneof '==' и '<='
 *
 */

namespace NTravel {

class TInvalidAgesException : public yexception {};

class TAges {
public:
    TAges() = default;

    static TAges Empty();

    // Ages
    static TAges FromAgesString(TStringBuf ages, bool allowTrailers = false);
    TString ToAgesString() const;

    // Occupancy
    static TAges FromOccupancyString(TStringBuf occupancy);
    TString ToOccupancyString() const;

    // From adults
    static TAges FromAdultCount(size_t adultCount);

    bool IsEmpty() const;

    size_t GetAdultCount() const;
    bool HasChildren() const;
    const TVector<size_t>& GetChildrenAges() const;
    size_t GetChildAgeMin() const;
    size_t GetChildAgeMax() const;

    bool operator ==(const TAges& rhs) const;
    bool operator !=(const TAges& rhs) const;

    size_t Hash() const;
    size_t GetAllocSize() const;
private:
    friend class TCapacity;
    size_t AdultCount = 0;
    TVector<size_t> ChildrenAges;

    TAges(size_t adultCount, TVector<size_t> childrenAges);
};

class TCapacity {
public:
    static TCapacity FromCapacityString(TStringBuf capacity);
    TString ToCapacityString() const;

    TAges DowncastToAges() const;
    static TCapacity FromAges(const TAges& ages);

    bool IsExact() const;
    size_t GetAdultCount() const;
    bool HasChildren() const;
    bool Matches(const TAges& ages) const;
    bool Matches(size_t adultsFrom, size_t adultsTo, bool allowChildren) const;

    bool operator ==(const TCapacity& rhs) const;
    size_t Hash() const;
    size_t GetAllocSize() const;
private:
    bool IsExactMatch = true;
    TAges Ages;
};

} // NTravel

template <>
struct THash<NTravel::TAges> {
    inline size_t operator() (const NTravel::TAges& v) const {
        return v.Hash();
    }
};

template <>
struct THash<NTravel::TCapacity> {
    inline size_t operator() (const NTravel::TCapacity& v) const {
        return v.Hash();
    }
};
