#include <travel/hotels/lib/cpp/ages/ages.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NTravel;

Y_UNIT_TEST_SUITE(TestAges) {
    TString TransformAges(const TString& ages, bool allowTrailers = false) {
        return TAges::FromAgesString(ages, allowTrailers).ToAgesString();
    }
    Y_UNIT_TEST(CheckAges) {
        UNIT_ASSERT_VALUES_EQUAL(TransformAges("88,0"), "0,88");
        UNIT_ASSERT_VALUES_EQUAL(TransformAges("88"), "88");
        UNIT_ASSERT_VALUES_EQUAL(TransformAges("1,88"), "1,88");
        UNIT_ASSERT_VALUES_EQUAL(TransformAges("1,2,88"), "1,2,88");
        UNIT_ASSERT_VALUES_EQUAL(TransformAges("1,2,3,88,88"), "1,2,3,88,88");
        UNIT_ASSERT_VALUES_EQUAL(TransformAges("88,88,88,88,88"), "88,88,88,88,88");
        UNIT_ASSERT_VALUES_EQUAL(TransformAges("1,2,3,4"), "1,2,3,4");
        UNIT_ASSERT_VALUES_EQUAL(TransformAges("17"), "17");
    }

    Y_UNIT_TEST(CheckAgesWithTrailers) {
        UNIT_ASSERT_VALUES_EQUAL(TransformAges("88.test,1.hello", true), "1,88");
    }

    Y_UNIT_TEST(CheckAgesSorting) {
        UNIT_ASSERT_VALUES_EQUAL(TransformAges("88,1"), "1,88");
        UNIT_ASSERT_VALUES_EQUAL(TransformAges("2,1,88"), "1,2,88");
        UNIT_ASSERT_VALUES_EQUAL(TransformAges("2,88,1"), "1,2,88");
        UNIT_ASSERT_VALUES_EQUAL(TransformAges("88,2,3,1,88"), "1,2,3,88,88");
        UNIT_ASSERT_VALUES_EQUAL(TransformAges("4,2,1,3"), "1,2,3,4");
    }
    Y_UNIT_TEST(CheckAgesWrong) {
        UNIT_ASSERT_EXCEPTION(TAges::FromAgesString(""), TInvalidAgesException);
        UNIT_ASSERT_EXCEPTION(TAges::FromAgesString("22"), TInvalidAgesException);
        UNIT_ASSERT_EXCEPTION(TAges::FromAgesString("ab"), TInvalidAgesException);
        UNIT_ASSERT_EXCEPTION(TAges::FromAgesString("ab13"), TInvalidAgesException);
        UNIT_ASSERT_EXCEPTION(TAges::FromAgesString("13a,4"), TInvalidAgesException);
        UNIT_ASSERT_EXCEPTION(TAges::FromAgesString("99"), TInvalidAgesException);
        UNIT_ASSERT_EXCEPTION(TAges::FromAgesString(",1"), TInvalidAgesException);
        UNIT_ASSERT_EXCEPTION(TAges::FromAgesString("1,,1"), TInvalidAgesException);
        UNIT_ASSERT_EXCEPTION(TAges::FromAgesString("88.test"), TInvalidAgesException);
    }

    TString TransformOccupancy(const TString& ages) {
        return TAges::FromOccupancyString(ages).ToOccupancyString();
    }
    Y_UNIT_TEST(CheckOccupancy) {
        UNIT_ASSERT_VALUES_EQUAL(TransformOccupancy("1"), "1");
        UNIT_ASSERT_VALUES_EQUAL(TransformOccupancy("1-1"), "1-1");
        UNIT_ASSERT_VALUES_EQUAL(TransformOccupancy("1-1,2"), "1-1,2");
        UNIT_ASSERT_VALUES_EQUAL(TransformOccupancy("2-1,2,3"), "2-1,2,3");
        UNIT_ASSERT_VALUES_EQUAL(TransformOccupancy("5"), "5");
        UNIT_ASSERT_VALUES_EQUAL(TransformOccupancy("0-1,2,3,4"), "0-1,2,3,4");
        UNIT_ASSERT_VALUES_EQUAL(TransformOccupancy("0-0,1,2,3"), "0-0,1,2,3");
        UNIT_ASSERT_VALUES_EQUAL(TransformOccupancy("0-17"), "0-17");
    }

    Y_UNIT_TEST(CheckOccupancySorting) {
        UNIT_ASSERT_VALUES_EQUAL(TransformOccupancy("2-3,2,1"), "2-1,2,3");
        UNIT_ASSERT_VALUES_EQUAL(TransformOccupancy("0-1,7,3"), "0-1,3,7");
    }
    Y_UNIT_TEST(CheckOccupancyWrong) {
        UNIT_ASSERT_EXCEPTION(TAges::FromOccupancyString(""), TInvalidAgesException);
        UNIT_ASSERT_EXCEPTION(TAges::FromOccupancyString("0"), TInvalidAgesException);
        UNIT_ASSERT_EXCEPTION(TAges::FromOccupancyString(""), TInvalidAgesException);
        UNIT_ASSERT_EXCEPTION(TAges::FromOccupancyString("2-2-3"), TInvalidAgesException);
        UNIT_ASSERT_EXCEPTION(TAges::FromOccupancyString("ab13"), TInvalidAgesException);
        UNIT_ASSERT_EXCEPTION(TAges::FromOccupancyString("13a,4"), TInvalidAgesException);
        UNIT_ASSERT_EXCEPTION(TAges::FromOccupancyString("-1"), TInvalidAgesException);
        UNIT_ASSERT_EXCEPTION(TAges::FromOccupancyString("1--1"), TInvalidAgesException);
    }

    TString TransformCapacity(const TString& ages) {
        return TCapacity::FromCapacityString(ages).ToCapacityString();
    }
    Y_UNIT_TEST(CheckCapacity) {
        UNIT_ASSERT_VALUES_EQUAL(TransformCapacity("==1"), "==1");
        UNIT_ASSERT_VALUES_EQUAL(TransformCapacity("<=1-1"), "<=1-1");
        UNIT_ASSERT_VALUES_EQUAL(TransformCapacity("<=1-1,2"), "<=1-1,2");
        UNIT_ASSERT_VALUES_EQUAL(TransformCapacity("<=2-1,2,3"), "<=2-1,2,3");
        UNIT_ASSERT_VALUES_EQUAL(TransformCapacity("==5"), "==5");
        UNIT_ASSERT_VALUES_EQUAL(TransformCapacity("==0-1,2,3,4"), "==0-1,2,3,4");
        UNIT_ASSERT_VALUES_EQUAL(TransformCapacity("<=0-17"), "<=0-17");
    }
    Y_UNIT_TEST(CheckCapacityWrong) {
        UNIT_ASSERT_EXCEPTION(TCapacity::FromCapacityString(""), TInvalidAgesException);
        UNIT_ASSERT_EXCEPTION(TCapacity::FromCapacityString("0"), TInvalidAgesException);
        UNIT_ASSERT_EXCEPTION(TCapacity::FromCapacityString(">=22"), TInvalidAgesException);
        UNIT_ASSERT_EXCEPTION(TCapacity::FromCapacityString("?"), TInvalidAgesException);
    }

    bool CapMatch(const TString& capacity, const TString& occupancy) {
        return TCapacity::FromCapacityString(capacity).Matches(TAges::FromOccupancyString(occupancy));
    }

    Y_UNIT_TEST(CheckCapacityMatchExact) {
        // exact
        UNIT_ASSERT(CapMatch("==2-2", "2-2"));
        UNIT_ASSERT(!CapMatch("==2-2", "2-3"));
        UNIT_ASSERT(!CapMatch("==2-2", "3-2"));
        UNIT_ASSERT(!CapMatch("==2-2", "1-2"));
        UNIT_ASSERT(!CapMatch("==2-2", "1"));
    }

    Y_UNIT_TEST(CheckCapacityMatchNotExact) {
        // <=
        UNIT_ASSERT(CapMatch("<=3", "1"));
        UNIT_ASSERT(CapMatch("<=3", "2"));
        UNIT_ASSERT(CapMatch("<=3", "3"));
        UNIT_ASSERT(!CapMatch("<=3", "4"));
        UNIT_ASSERT(CapMatch("<=3", "1-1"));
        UNIT_ASSERT(CapMatch("<=3", "1-1,2"));
        UNIT_ASSERT(!CapMatch("<=3", "1-1,2,3"));
        UNIT_ASSERT(CapMatch("<=3", "1-1,1"));
        UNIT_ASSERT(CapMatch("<=3", "1-1,2"));
        UNIT_ASSERT(CapMatch("<=3", "1-1,12"));
    }

    Y_UNIT_TEST(CheckCapacityMatchNotExactWithChildren) {
        // <= + Children
        UNIT_ASSERT(CapMatch("<=1-1,2,3", "1"));
        UNIT_ASSERT(CapMatch("<=1-1,2,3", "1-1,2"));
        UNIT_ASSERT(CapMatch("<=1-1,2,3", "1-1,2,3"));
        UNIT_ASSERT(CapMatch("<=1-1,2,3", "1-2,3"));
        UNIT_ASSERT(!CapMatch("<=1-1,2,3", "1-2,3,4"));
        UNIT_ASSERT(CapMatch("<=1-1,2,3", "0-2,3,4"));
        UNIT_ASSERT(!CapMatch("<=1-1,2,3", "0-2,3,4,5"));
        UNIT_ASSERT(CapMatch("<=1-9,9,9", "0-2,3,4"));
        UNIT_ASSERT(CapMatch("<=1-9,9,9", "1-2,3,4"));
        UNIT_ASSERT(CapMatch("<=1-9,9,9", "1-2,3,9"));
        UNIT_ASSERT(!CapMatch("<=1-9,9,9", "1-2,3,10"));
        UNIT_ASSERT(CapMatch("<=1-9,9,9", "0-2,3,10"));
        UNIT_ASSERT(!CapMatch("<=1-9,9,9", "0-2,10,10"));
    }
};
