#include "data.h"

#include <travel/hotels/lib/cpp/util/sizes.h>
#include <travel/hotels/lib/cpp/protobuf/tools.h>
#include "travel/hotels/proto/data_config/google_hotels_black_list_item.pb.h"
#include "travel/hotels/proto/data_config/hotel_wizard_ban.pb.h"
#include "travel/hotels/proto/data_config/operator.pb.h"
#include "travel/hotels/proto/data_config/partner.pb.h"
#include "travel/hotels/proto/data_config/offercache_client.pb.h"
#include <travel/hotels/proto/data_config/rate_plans/travelline_rate_plan_info.pb.h>
#include <travel/hotels/proto/data_config/rate_plans/dolphin_list_record.pb.h>
#include <travel/hotels/proto/data_config/rate_plans/bnovo_rate_plan_info.pb.h>
#include <travel/hotels/proto/data_config/searchkey_restrictions.pb.h>
#include <travel/hotels/proto/data_config/promo/black_friday_2021_hotel.pb.h>

#include <util/digest/multi.h>
#include <util/string/builder.h>
#include <util/string/cast.h>
#include <util/stream/str.h>

namespace NTravel {

//----------------

void TPriceValRange::Update(TPriceVal priceVal) {
    if (IsValid) {
        MinPriceVal = Min(MinPriceVal, priceVal);
        MaxPriceVal = Max(MaxPriceVal, priceVal);
    } else {
        MinPriceVal = priceVal;
        MaxPriceVal = priceVal;
        IsValid = true;
    }
}

TPriceWithCurrency::TPriceWithCurrency(NTravelProto::ECurrency currency, TPriceVal value)
    : Currency(currency)
    , Value(value)
{
}


TPriceWithCurrency TPriceWithCurrency::FromProto(const NTravelProto::TPrice& proto) {
    if (proto.GetPrecision() != 0) {
        throw yexception() << "Non-zero precision in price value " << proto.GetAmount();
    }
    return TPriceWithCurrency(proto.GetCurrency(), proto.GetAmount());
}

void TPriceWithCurrency::ToProto(NTravelProto::TPrice* proto) const {
    proto->SetCurrency(Currency);
    proto->SetAmount(Value);
    proto->SetPrecision(0);
}

TString THotelId::ToCompositeString() const {
    return ::ToString(PartnerId) + ".'" + OriginalId + "'";
}

THotelId THotelId::FromProto(const NTravelProto::THotelId& proto) {
    THotelId res;
    res.PartnerId = proto.GetPartnerId();
    res.OriginalId = proto.GetOriginalId();
    return res;
}

void THotelId::ToProto(NTravelProto::THotelId* proto) const {
    proto->SetPartnerId((NTravelProto::EPartnerId)PartnerId);
    proto->SetOriginalId(OriginalId);
}

bool THotelId::operator<(const THotelId& rhs) const {
    if (PartnerId != rhs.PartnerId) {
        return PartnerId < rhs.PartnerId;
    }
    return OriginalId < rhs.OriginalId;
}

bool THotelId::operator==(const THotelId& rhs) const {
    return PartnerId == rhs.PartnerId && OriginalId == rhs.OriginalId;
}

bool THotelId::operator!=(const THotelId& rhs) const {
    return !operator ==(rhs);
}

size_t THotelId::Hash() const {
    return MultiHash(PartnerId, OriginalId);
}

size_t THotelId::GetAllocSize() const {
    return TTotalByteSize<TString>()(OriginalId) - sizeof(OriginalId);
}

size_t THotelId::CalcTotalByteSize() const {
    return sizeof(THotelId) + TTotalByteSize<TString>()(OriginalId) - sizeof(OriginalId);
}

//----------------


bool TPreKey::operator == (const TPreKey& rhs) const {
    return Currency == rhs.Currency && HotelId == rhs.HotelId;
}

bool TPreKey::operator != (const TPreKey& rhs) const {
    return !operator ==(rhs);
}

size_t TPreKey::Hash() const {
    return MultiHash(Currency, HotelId);
}

size_t TPreKey::GetAllocSize() const {
    return HotelId.GetAllocSize();
}

size_t TPreKey::CalcTotalByteSize() const {
    return sizeof(TPreKey) + HotelId.GetAllocSize();
}


//----------------
bool TSearchSubKey::IsComplete() const {
    return (Date != NOrdinalDate::g_DateZero) && (Nights != g_NightsZero) && !Ages.IsEmpty();
}

void TSearchSubKey::Merge(const TSearchSubKey& other) {
    if (Date == NOrdinalDate::g_DateZero) {
        Date = other.Date;
    }
    if (Nights == g_NightsZero) {
        Nights = other.Nights;
    }
    if (Ages.IsEmpty()) {
        Ages = other.Ages;
    }
}

bool TSearchSubKey::operator == (const TSearchSubKey& rhs) const {
    return Ages == rhs.Ages && Date == rhs.Date && Nights == rhs.Nights;
}

size_t TSearchSubKey::Hash() const {
    return MultiHash(Ages, Date, Nights);
}

size_t TSearchSubKey::GetAllocSize() const {
    return Ages.GetAllocSize();
}

//----------------

bool TSearchKey::operator == (const TSearchKey& rhs) const {
    return PreKey == rhs.PreKey && SubKey == rhs.SubKey;
}

size_t TSearchKey::Hash() const {
    return MultiHash(PreKey, SubKey);
}

size_t TSearchKey::GetAllocSize() const {
    return PreKey.GetAllocSize() + SubKey.GetAllocSize();
}

TSearchKey TSearchKey::FromRequest(const NTravelProto::TSearchOffersReq& req) {
    TSearchKey key;
    key.PreKey.Currency = req.GetCurrency();
    key.PreKey.HotelId = THotelId::FromProto(req.GetHotelId());
    key.SubKey.Date = NOrdinalDate::FromString(req.GetCheckInDate());
    key.SubKey.Nights = NOrdinalDate::FromString(req.GetCheckOutDate()) - key.SubKey.Date;
    key.SubKey.Ages = TAges::FromOccupancyString(req.GetOccupancy());
    return key;
}
//----------------
bool TSearcherRequestKey::operator == (const TSearcherRequestKey& rhs) const {
    return SearchKey == rhs.SearchKey && RequestClass == rhs.RequestClass;
}

size_t TSearcherRequestKey::Hash() const {
    return MultiHash(SearchKey, RequestClass);
}

size_t TSearcherRequestKey::GetAllocSize() const {
    return SearchKey.GetAllocSize();
}

TSearcherRequestKey TSearcherRequestKey::FromRequest(const NTravelProto::TSearchOffersReq& req) {
    TSearcherRequestKey key;
    key.SearchKey = TSearchKey::FromRequest(req);
    key.RequestClass = req.GetRequestClass();
    return key;
}
//----------------

TString GetSearchOffersRpcReqIds(const NTravelProto::TSearchOffersRpcReq& req) {
    TStringBuilder requestIds;
    for (const auto& subReq: req.GetSubrequest()) {
        if (requestIds) {
            requestIds << ", ";
        }
        requestIds << subReq.GetId();
    }
    return requestIds;
}

const NYT::TNode& GetColumn(const NYT::TNode& row, TStringBuf name1, TStringBuf name2) {
    const NYT::TNode& node = row[name1];
    if (!node.IsUndefined()) {
        return node;
    }
    return row[name2];
}

TPermalink ExtractPermalinkFromYtRow(const NYT::TNode& row) {
    const NYT::TNode& permalinkNode = GetColumn(row, TStringBuf("permalink"), TStringBuf("Permalink"));
    if (permalinkNode.IsInt64() || permalinkNode.IsUint64()) {
        return permalinkNode.IntCast<ui64>();
    }
    return 0;
}

EPartnerId ExtractPartnerIdFromYtRow(const NYT::TNode& row) {
    const NYT::TNode& partnerIdIntNode = GetColumn(row, TStringBuf("partnerid"), TStringBuf("PartnerIdInt"));
    if (!partnerIdIntNode.IsUndefined()) {
        return (EPartnerId)partnerIdIntNode.IntCast<std::underlying_type_t<EPartnerId>>();
    }

    const NYT::TNode& partnerIdNode = row["PartnerId"];
    if (partnerIdNode.IsString()) {
        EPartnerId partnerId;
        if (NTravelProto::EPartnerId_Parse(partnerIdNode.AsString(), &partnerId)) {
            return partnerId;
        }
    }
    return NTravelProto::PI_UNUSED;
}

TString    ExtractOriginalIdFromYtRow(const NYT::TNode& row) {
    const NYT::TNode& originalIdNode = GetColumn(row, TStringBuf("originalid"), TStringBuf("OriginalId"));
    if (originalIdNode.IsString()) {
        return originalIdNode.AsString();
    }
    return TString();
}

bool HasBreakfast(NTravelProto::EPansionType pt) {
    // Should be same list as here https://a.yandex-team.ru/arc/trunk/arcadia/travel/hotels/tools/build_hotel_traits/queries/hotel_traits.yql
    // See TRAVELBACK-503, TRAVELBACK-1612 for details
    return pt != NTravelProto::EPansionType::PT_UNKNOWN &&
           pt != NTravelProto::EPansionType::PT_RO &&
           pt != NTravelProto::EPansionType::PT_BD;
}

bool HasBreakfastDinner(NTravelProto::EPansionType pt) {
    // Should be same list as here https://a.yandex-team.ru/arc/trunk/arcadia/travel/hotels/tools/build_hotel_traits/queries/hotel_traits.yql
    return IsIn({
                    NTravelProto::EPansionType::PT_AI,
                    NTravelProto::EPansionType::PT_FB,
                    NTravelProto::EPansionType::PT_HB,
                    NTravelProto::EPansionType::PT_UAI,
                    NTravelProto::EPansionType::PT_LAI
                }, pt);
}

bool HasBreakfastLunchDinner(NTravelProto::EPansionType pt) {
    // Should be same list as here https://a.yandex-team.ru/arc/trunk/arcadia/travel/hotels/tools/build_hotel_traits/queries/hotel_traits.yql
    return IsIn({
                    NTravelProto::EPansionType::PT_AI,
                    NTravelProto::EPansionType::PT_FB,
                    NTravelProto::EPansionType::PT_UAI,
                    NTravelProto::EPansionType::PT_LAI
                }, pt);
}

bool HasAllInclusive(NTravelProto::EPansionType pt) {
    // Should be same list as here https://a.yandex-team.ru/arc/trunk/arcadia/travel/hotels/tools/build_hotel_traits/queries/hotel_traits.yql
    return IsIn({
                    NTravelProto::EPansionType::PT_AI,
                    NTravelProto::EPansionType::PT_UAI,
                    NTravelProto::EPansionType::PT_LAI
                }, pt);
}

bool HasNoPansion(NTravelProto::EPansionType pt) {
    // Should be same list as here https://a.yandex-team.ru/arc/trunk/arcadia/travel/hotels/tools/build_hotel_traits/queries/hotel_traits.yql
    return IsIn({
                    NTravelProto::EPansionType::PT_UNKNOWN,
                    NTravelProto::EPansionType::PT_RO,
                }, pt);
}

}// namespace NTravel

//---------------------------------------------------------------------------------------

template <>
void Out<NTravel::THotelId>(IOutputStream& out, const NTravel::THotelId& id) {
    out << id.ToCompositeString();
}

template <>
void Out<NTravel::TSearchSubKey>(IOutputStream& out, const NTravel::TSearchSubKey& key) {
    out << "Date: " << NTravel::NOrdinalDate::ToString(key.Date)
        << ", Nights: " << (int)key.Nights
        << ", Ages: " << key.Ages.ToAgesString();
}

template <>
void Out<NTravel::TSearchKey>(IOutputStream& out, const NTravel::TSearchKey& key) {
    out << "HotelId: " << key.PreKey << ", " << key.SubKey;
}

template <>
void Out<NTravelProto::ECurrency>(IOutputStream& out, NTravelProto::ECurrency c) {
    out << NTravelProto::ECurrency_Name(c);
}


template <>
void Out<NTravel::TPreKey>(IOutputStream& out, const NTravel::TPreKey& key) {
    out << key.HotelId << "/" << key.Currency;
}

template <>
void Out<NTravel::TSearcherRequestKey>(IOutputStream& out, const NTravel::TSearcherRequestKey& key) {
    out << key.SearchKey << "(" << NTravelProto::ERequestClass_Name(key.RequestClass) << ")";
}

template <>
void Out<NTravelProto::EOperatorId>(IOutputStream& out, NTravelProto::EOperatorId id) {
    out << NTravelProto::EOperatorId_Name(id);
}

template <>
void Out<NTravelProto::EPartnerId>(IOutputStream& out, NTravelProto::EPartnerId id) {
    out << NTravelProto::EPartnerId_Name(id);
}

template <>
void Out<NTravelProto::EPansionType>(IOutputStream& out, NTravelProto::EPansionType pansionType) {
    out << NTravelProto::EPansionType_Name(pansionType);
}

template <>
void Out<NTravel::TOfferCacheClientKey>(IOutputStream& out, const NTravel::TOfferCacheClientKey& id) {
    out << id.first << "/" << id.second;
}

template <>
void Out<NTravel::TTravelLineRatePlanKey>(IOutputStream& out, const NTravel::TTravelLineRatePlanKey& id) {
    out << std::get<0>(id) << "/" << std::get<1>(id);
}

template <>
void Out<NTravel::TBNovoRatePlanKey>(IOutputStream& out, const NTravel::TBNovoRatePlanKey& id) {
    out << std::get<0>(id) << "/" << std::get<1>(id);
}

namespace NTravel::NProtobuf {

template <>
NTravelProto::EOperatorId GetIdentifier<NTravelProto::EOperatorId, NTravelProto::NConfig::TOperator>(const NTravelProto::NConfig::TOperator& data) {
    return data.GetOperatorId();
}

template <>
NTravelProto::EPartnerId GetIdentifier<NTravelProto::EPartnerId, NTravelProto::NConfig::TPartner>(const NTravelProto::NConfig::TPartner& data) {
    return data.GetPartnerId();
}

template <>
NTravel::TOfferCacheClientKey GetIdentifier<NTravel::TOfferCacheClientKey, NTravelProto::NConfig::TOfferCacheClient>(const NTravelProto::NConfig::TOfferCacheClient& data) {
    return std::make_pair(data.GetGeoOrigin(), data.GetGeoClientId());
}

template <>
NTravel::TTravelLineRatePlanKey GetIdentifier<NTravel::TTravelLineRatePlanKey, NTravelProto::NConfig::TTravellineRatePlanInfo>(const NTravelProto::NConfig::TTravellineRatePlanInfo& data) {
    return std::make_tuple(data.GetHotelCode(), data.GetRatePlanCode());
}

template <>
ui64 GetIdentifier<ui64, NTravelProto::NConfig::TDolphinListRecord>(const NTravelProto::NConfig::TDolphinListRecord& data) {
    return data.GetKey();
}

template <>
NTravel::TBNovoRatePlanKey GetIdentifier<NTravel::TBNovoRatePlanKey, NTravelProto::NConfig::TBNovoRatePlanInfo>(const NTravelProto::NConfig::TBNovoRatePlanInfo& data) {
    return std::make_tuple(data.GetAccountId(), data.GetRatePlanId());
}

template <>
NTravel::TSearchKeyRestrictionsKey GetIdentifier<NTravel::TSearchKeyRestrictionsKey, NTravelProto::NConfig::TSearchKeyRestrictions>(const NTravelProto::NConfig::TSearchKeyRestrictions& data) {
    NTravel::THotelId res;
    res.PartnerId = data.GetPartnerId();
    res.OriginalId = data.GetOriginalId();
    return res;
}

template <>
TPermalink GetIdentifier<TPermalink, NTravelProto::NConfig::THotelWizardBan>(const NTravelProto::NConfig::THotelWizardBan& data) {
    return data.GetPermalink();
}

template <>
THotelId GetIdentifier<THotelId, NTravelProto::NConfig::TBlackFriday2021Hotel>(const NTravelProto::NConfig::TBlackFriday2021Hotel& data) {
    THotelId h;
    h.PartnerId = data.GetPartnerId();
    h.OriginalId = data.GetOriginalId();
    return h;
}

template <>
TString GetIdentifier<TString, NTravelProto::NConfig::TGoogleHotelsBlackListItem>(const NTravelProto::NConfig::TGoogleHotelsBlackListItem& data) {
    return data.GetSlug();
}


}//NTravel::NProtobuf



//---------------------------------------------------------------------------------------
