#pragma once

#include "travel/hotels/proto2/hotels.pb.h"

#include <travel/hotels/lib/cpp/ordinal_date/ordinal_date.h>
#include <travel/hotels/lib/cpp/ages/ages.h>

#include <library/cpp/yson/node/node.h>

#include <util/generic/vector.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/ptr.h>
#include <util/generic/maybe.h>

namespace NTravel {

using TNights = ui32;
using EPartnerId = NTravelProto::EPartnerId;
using EOperatorId = NTravelProto::EOperatorId;
using TPriceVal = ui32;
using TPermalink = ui64;
using TOfferIdHash = ui32;

constexpr TNights g_NightsZero = 0;

struct TPriceValRange {
    bool IsValid = false;
    TPriceVal MinPriceVal;
    TPriceVal MaxPriceVal;

    void Update(TPriceVal priceVal);
};

struct TPriceWithCurrency {
    TPriceWithCurrency(NTravelProto::ECurrency currency = NTravelProto::C_RUB, TPriceVal value = 0);

    static TPriceWithCurrency FromProto(const NTravelProto::TPrice& proto);
    void ToProto(NTravelProto::TPrice* proto) const;

    NTravelProto::ECurrency Currency;
    TPriceVal Value;
};

enum class EFreeCancellationType : ui8 {
    Unknown,
    No,
    Yes
};

struct THotelId {
    EPartnerId PartnerId;  // идентификатор партнёра
    TString    OriginalId; // Идентификатор отеля у партнёра

    TString ToCompositeString() const;

    static THotelId FromProto(const NTravelProto::THotelId& proto);
    void ToProto(NTravelProto::THotelId* proto) const;

    bool operator <(const THotelId& rhs) const;
    bool operator==(const THotelId& rhs) const;
    bool operator!=(const THotelId& rhs) const;
    size_t Hash() const;
    size_t GetAllocSize() const;
    size_t CalcTotalByteSize() const;
};

}// namespace NTravel

template <>
struct THash<NTravel::THotelId> {
    inline size_t operator() (const NTravel::THotelId& v) const { return v.Hash(); }
};


namespace NTravel {

struct TComplexHotelId {
    THashSet<THotelId> HotelIds;
};

struct TPreKey {
    THotelId HotelId;
    NTravelProto::ECurrency Currency;
    bool operator == (const TPreKey& rhs) const;
    bool operator != (const TPreKey& rhs) const;
    size_t Hash() const;
    size_t GetAllocSize() const;
    size_t CalcTotalByteSize() const;
};

//----------------------------------

struct TSearchSubKey {
    TAges                      Ages;
    NOrdinalDate::TOrdinalDate Date   = NOrdinalDate::g_DateZero;
    TNights                    Nights = g_NightsZero;

    bool IsComplete() const;
    void Merge(const TSearchSubKey& other);

    bool operator == (const TSearchSubKey& rhs) const;
    size_t Hash() const;
    size_t GetAllocSize() const;
};

struct TSearchKey  {
    TPreKey PreKey;
    TSearchSubKey SubKey;
    bool operator == (const TSearchKey& rhs) const;
    size_t Hash() const;
    size_t GetAllocSize() const;

    static TSearchKey FromRequest(const NTravelProto::TSearchOffersReq& req);
};

struct TSearcherRequestKey {
    TSearchKey SearchKey;
    NTravelProto::ERequestClass RequestClass;

    bool operator == (const TSearcherRequestKey& rhs) const;
    size_t Hash() const;
    size_t GetAllocSize() const;

    static TSearcherRequestKey FromRequest(const NTravelProto::TSearchOffersReq& req);
};

TString GetSearchOffersRpcReqIds(const NTravelProto::TSearchOffersRpcReq& req);

using TOfferCacheClientKey = std::pair<TString/*Origin*/, TString/*ClientId*/>;
using TTravelLineRatePlanKey = std::tuple<TString/*HotelCode*/, TString/*RatePlanCode*/>;
using TBNovoRatePlanKey = std::tuple<i64/*AccountId*/, i64/*RatePlanId*/>;
using TSearchKeyRestrictionsKey = THotelId;

TPermalink ExtractPermalinkFromYtRow(const NYT::TNode& row);
EPartnerId ExtractPartnerIdFromYtRow(const NYT::TNode& row);
TString    ExtractOriginalIdFromYtRow(const NYT::TNode& row);
bool HasBreakfast(NTravelProto::EPansionType pt);
bool HasBreakfastDinner(NTravelProto::EPansionType pt);
bool HasBreakfastLunchDinner(NTravelProto::EPansionType pt);
bool HasAllInclusive(NTravelProto::EPansionType pt);
bool HasNoPansion(NTravelProto::EPansionType pt);

}// namespace NTravel

//----------------------------------

template <>
struct THash<NTravel::TSearchSubKey> {
    inline size_t operator() (const NTravel::TSearchSubKey& v) const { return v.Hash(); }
};

template <>
struct THash<NTravel::TPreKey> {
    inline size_t operator() (const NTravel::TPreKey& v) const { return v.Hash(); }
};


template <>
struct THash<NTravel::TSearchKey> {
    inline size_t operator() (const NTravel::TSearchKey& v) const { return v.Hash(); }
};

template <>
struct THash<NTravel::TSearcherRequestKey> {
    inline size_t operator() (const NTravel::TSearcherRequestKey& v) const { return v.Hash(); }
};
