#include "grpc_async_client.h"

#include <util/generic/singleton.h>

namespace NTravel {
namespace NGrpc {

TClientJobWatcher::TClientJobWatcher()
    : LastJobId_(0)
    , LastClientId_(0)
{
    Thread_ = SystemThreadFactory()->Run([this](){ DoExecute();});
}

TClientJobWatcher::~TClientJobWatcher() {
    INFO_LOG << "Shutting down grpc watcher..." << Endl;
    if (StopFlag_ || !Thread_) {
        return;
    }
    StopFlag_.Set();
    with_lock (Lock_) {
        CompletionQueue_.Shutdown();
    }
    Thread_->Join();
    Thread_ .Reset();
    INFO_LOG << "Shutting down grpc watcher DONE" << Endl;
}

bool TClientJobWatcher::RegisterJob(TClientId clientId, TClientJobBaseRef job, TFunc func) {
    size_t jobId;
    with_lock (Lock_) {
        if (StopFlag_) {
            return false;
        }
        jobId = ++LastJobId_;
        Jobs_[jobId] = std::make_pair(clientId, job);
        ClientJobs_[clientId].insert(jobId);
    }
    func(CompletionQueue_, jobId);
    return true;
}

TClientId TClientJobWatcher::GenerateClientId() {
    with_lock (Lock_) {
        return ++LastClientId_;
    }
}

void TClientJobWatcher::CancelJobsForClient(TClientId clientId) {
    INFO_LOG << "Cancel jobs for client " << clientId << Endl;
    TVector<TClientJobBaseRef> jobsToCancel;
    with_lock (Lock_) {
        auto& jobIds = ClientJobs_[clientId];
        jobsToCancel.reserve(jobIds.size());
        for (auto jobId: jobIds) {
            auto it = Jobs_.find(jobId);
            if (it != Jobs_.end()) {
                jobsToCancel.push_back(it->second.second);
            }
        }
    }
    for (const TClientJobBaseRef& job: jobsToCancel) {
        job->OnCancel();
    }
}

void TClientJobWatcher::DoExecute() {
    while (!StopFlag_) {
        void* tag;
        bool ok;
        if (!CompletionQueue_.Next(&tag, &ok)) {
            continue;
        }
        size_t jobId = (size_t)tag;
        TClientJobBaseRef job;
        with_lock (Lock_) {
            auto it = Jobs_.find(jobId);
            if (it == Jobs_.end()) {
                ERROR_LOG << "gRPC client job not found by id " << jobId << Endl;
                continue;
            }
            TClientId clientId = it->second.first;
            job = it->second.second;
            Jobs_.erase(it);
            ClientJobs_[clientId].erase(jobId);
        }
        job->OnComplete(ok);
    }
}

TClientJobWatcher& TClientJobWatcher::Instance() {
    return *Singleton<TClientJobWatcher>();
}

}// Namespace NGrpc
}// Namespace NTravel
