#pragma once

#include <grpc/grpc.h>
#include <grpc++/channel.h>
#include <grpc++/client_context.h>
#include <grpc++/completion_queue.h>
#include <grpc++/create_channel.h>
#include <grpc++/support/async_stream.h>
#include <grpc++/support/async_unary_call.h>

#include <travel/hotels/lib/cpp/util/profiletimer.h>
#include <travel/hotels/lib/cpp/util/flag.h>
#include <travel/hotels/proto/app_config/grpc.pb.h>

#include <library/cpp/logger/global/global.h>

#include <util/generic/guid.h>
#include <util/generic/ptr.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/datetime/base.h>
#include <library/cpp/deprecated/atomic/atomic.h>
#include <util/system/mutex.h>
#include <util/system/hostname.h>
#include <util/string/builder.h>
#include <util/thread/factory.h>

#include <functional>
#include <memory>

namespace NTravel {
namespace NGrpc {

struct TClientMetadata {
    TString CallId;
    TString ForwardedFor;
};

struct TClientJobBase : public TThrRefBase  {
    virtual void Start() = 0;
    virtual void OnCancel() = 0;
    virtual void OnComplete(bool ok) = 0;
};
using TClientJobBaseRef = TIntrusivePtr<TClientJobBase>;

using TClientId = size_t;

class TClientJobWatcher {
public:
    using TFunc = std::function<void (grpc::CompletionQueue& cq, size_t jobId)>;

    TClientJobWatcher();
    ~TClientJobWatcher();

    bool RegisterJob(TClientId clientId, TClientJobBaseRef job, TFunc func);
    TClientId GenerateClientId();
    void CancelJobsForClient(TClientId clientId);

    static TClientJobWatcher& Instance();
private:

    TAutoPtr<IThreadFactory::IThread> Thread_;
    TAtomicFlag StopFlag_;

    TMutex Lock_;
    size_t LastJobId_;
    TClientId LastClientId_;
    grpc::CompletionQueue CompletionQueue_;
    THashMap<size_t, std::pair<TClientId, TClientJobBaseRef>> Jobs_;
    THashMap<TClientId, THashSet<size_t>> ClientJobs_;

    void DoExecute();
};

template <class TGrpcService>
class TAsyncClient {
private:
    static gpr_timespec MakeDeadline(TInstant t) {
        gpr_timespec dl;
        dl.clock_type = GPR_CLOCK_REALTIME;
        dl.tv_sec = t.Seconds();
        dl.tv_nsec = t.NanoSecondsOfSecond();
        return dl;
    }

    template <class TGrpcRequest, class TGrpcResponse, std::unique_ptr<::grpc::ClientAsyncResponseReader<TGrpcResponse>> (TGrpcService::Stub::*StartFunc)(::grpc::ClientContext* context, const TGrpcRequest& request, ::grpc::CompletionQueue* cq)>
    struct TClientJob: public TClientJobBase {
        using TOnResponse = std::function<void (const TString& error, const TString& remoteFQDN, const TGrpcResponse&)>;

        TAsyncClient&       Client;
        const TDuration     Timeout;
        const size_t        Attempts;
        const size_t        CurrentAttempt;
        const TInstant      Started;
        const TGrpcRequest  Req;
        const TOnResponse   OnResponse;
        const TProfileTimer LifespanTimer;

        TAtomicFlag         IsComplete;
        TClientMetadata     Meta;
        grpc::ClientContext Context;
        grpc::Status        Status;
        TGrpcResponse       Resp;

        TClientJob(TAsyncClient& client, const TGrpcRequest& req, TOnResponse onResponse, const TClientMetadata& meta, TDuration timeout, size_t attempts, size_t currentAttempt)
            : Client(client)
            , Timeout(timeout)
            , Attempts(attempts)
            , CurrentAttempt(currentAttempt)
            , Started(::Now())
            , Req(req)
            , OnResponse(onResponse)
            , Meta(meta)
        {
            if (!Meta.CallId) {
                Meta.CallId = CreateGuidAsString();
            }
            Context.set_deadline(MakeDeadline(Started + Timeout));
            if (Meta.ForwardedFor) {
                Context.AddMetadata("ya-grpc-forwarded-for", Meta.ForwardedFor);
            }
            Context.AddMetadata("ya-grpc-call-id", Meta.CallId);
            Context.AddMetadata("ya-grpc-started-at", ToString(Started));
            Context.AddMetadata("ya-grpc-fqdn", FQDNHostName());
        }

        TString GetServerMetadata(const grpc::string_ref& name, const TString& defValue) {
            auto it = Context.GetServerInitialMetadata().find(name);
            if (it == Context.GetServerInitialMetadata().end()) {
                return defValue;
            }
            return TString(it->second.data(), it->second.size());
        }

        void Start() override {
            DEBUG_LOG << "Starting gRPC call " << Meta.CallId << ", timeout " << Timeout  << ", attempt " << CurrentAttempt << "/" << Attempts << Endl;
            TClientJobWatcher::Instance().RegisterJob(Client.ClientId_, this, [this](grpc::CompletionQueue& cq, size_t jobId) {
                std::unique_ptr<grpc::ClientAsyncResponseReader<TGrpcResponse>> rpc = (Client.Stub_.get()->*StartFunc)(&Context, Req, &cq);
                rpc->Finish(&Resp, &Status, (void*)jobId);
            });
        }

        void OnCancel() override {
            if (!IsComplete.TrySet()) {
                return;
            }
            OnResponse("Cancelled", "", TGrpcResponse());
        }

        void OnComplete(bool ok) override {
            if (!IsComplete.TrySet()) {
                return;
            }
            auto callDuration = LifespanTimer.Get();
            TString remoteFQDN = GetServerMetadata("ya-grpc-fqdn", TString(Context.peer()));
            TStringBuilder error;
            if (!ok) {
                error << "Request cancelled";
            } else if (!Status.ok()) {
                error << "gRPC error code " << (int)Status.error_code() << ", message '"  << Status.error_message() << "'";
            }
            if (error) {
                ERROR_LOG << "Failed gRPC call " << Meta.CallId << " in " << callDuration << ", remote FQDN '" << remoteFQDN << "'" << ", error " << error << Endl;
                error << ". Request call id " << Meta.CallId << ", started at " << Started << ", which is " << callDuration << " ago";
            } else {
                DEBUG_LOG << "Completed gRPC call " << Meta.CallId << " in " << callDuration << ", remote FQDN '" << remoteFQDN << "'" << Endl;
            }
            if (error && (CurrentAttempt < Attempts)) {
                // TODO wait before retry?
                Client.Request<TGrpcRequest, TGrpcResponse, StartFunc>(Req, Meta, OnResponse, Timeout, Attempts, CurrentAttempt + 1);
                return;
            }
            try {
                OnResponse(error, remoteFQDN, Resp);
            } catch (...) {
                ERROR_LOG << "Exception during handling gRPC response for call " << Meta.CallId << ": " << CurrentExceptionMessage() << Endl;
            }
        }
    };
public:
    template <class TGrpcResponse>
    using TOnResponse = std::function<void (const TString& error, const TString& remoteFQDN, const TGrpcResponse&)>;

    TAsyncClient(const NTravelProto::NAppConfig::TConfigGrpcClient& cfg)
        : DefaultTimeout_(TDuration::Seconds(cfg.GetTimeoutSec()))
        , DefaultAttempts_(cfg.GetAttempts())
        , ClientId_(TClientJobWatcher::Instance().GenerateClientId())
        , Channel_(grpc::CreateChannel(cfg.GetAddress(), grpc::InsecureChannelCredentials()))
        , Stub_(TGrpcService::NewStub(Channel_))
    {
    }

    ~TAsyncClient() {
        Y_VERIFY(IsShuttingDown_);
    }

    // PLZ Call Shutdown() before ~dtor, otherwise it is possible to get unexpected job callbacks during ~dtor
    void Shutdown() {
        IsShuttingDown_.Set();
        TClientJobWatcher::Instance().CancelJobsForClient(ClientId_);
    }

    template <class TGrpcRequest, class TGrpcResponse, std::unique_ptr<::grpc::ClientAsyncResponseReader<TGrpcResponse>> (TGrpcService::Stub::*StartFunc)(::grpc::ClientContext* context, const TGrpcRequest& request, ::grpc::CompletionQueue* cq)>
    void Request(const TGrpcRequest& req, const TClientMetadata& meta, TOnResponse<TGrpcResponse> onResponse,  TDuration timeout = TDuration::Zero(), size_t attempts = 0, size_t currentAttempt = 1) {
        if (IsShuttingDown_) {
            onResponse("Shutting down", "", TGrpcResponse());
            return;
        }
        TClientJobBaseRef job = new TClientJob<TGrpcRequest, TGrpcResponse, StartFunc>(*this, req, onResponse, meta,
                                                                                       timeout == TDuration::Zero() ? DefaultTimeout_ : timeout,
                                                                                       attempts == 0 ? DefaultAttempts_ : attempts,
                                                                                       currentAttempt);
        job->Start();
    }
private:
    const TDuration DefaultTimeout_;
    const size_t DefaultAttempts_;
    const TClientId ClientId_;

    TAtomicFlag IsShuttingDown_;
    std::shared_ptr<grpc::Channel> Channel_;
    std::unique_ptr<typename TGrpcService::Stub> Stub_;
};

}// Namespace NGrpc
}// Namespace NTravel
