#include "http_service.h"

#include <travel/hotels/lib/cpp/util/profiletimer.h>

#include <library/cpp/logger/global/global.h>

#include <util/string/strip.h>
#include <util/string/ascii.h>
#include <util/string/cast.h>
#include <util/string/builder.h>

namespace NTravel {
namespace NHttp {

using TConnectionRef = TIntrusivePtr<TConnection>;

class TConnection : public TThrRefBase, public TClientRequest {
public:
    TConnection(TServer& server)
        : Server_(server)
        , RequestName_("Not parsed")
    {}
private:
    /*
     * There is THolder one stack frame above.
     * It will destroy current connection if this method return true.
     * If this method returns false, nobody owns current connection and we keep references in callbacks.
     */
    bool Reply(void*) override {
        try {
            Y_ENSURE(Request_.Method().empty(), "Request should be created only once");
            TParsedHttpFull parsed(Input().FirstLine());
            Request_.Init(parsed, this);
            RequestName_ = Request_.Method() + " " + parsed.Request;

            for (auto& callback: Server_.Callbacks_) {
                if (callback.AcceptsRequest(Request_)) {
                    TConnectionRef self(this);
                    try {
                        callback.HandleRequest(Request_, [self] (const TResponse& response) {
                            self->SendResponse(response);
                        });
                        // async response, return false and keep self reference in callbacks,
                        // now we own this connection
                        return false;
                    }
                    catch (...) {
                        // error occured, release self ref as connection will be released up to the stack
                        Y_UNUSED(self.Release());
                        throw;
                    }
                }
            }
            Server_.Counters_.NHttpNotFound.Inc();
            SendResponse(TResponse::CreateText("Handler not found for '" + RequestName_ + "'\n", HTTP_NOT_FOUND));
        }
        catch (...) {
            SendResponse(TResponse::CreateText(CurrentExceptionMessage() + "\n", HTTP_INTERNAL_SERVER_ERROR));
        }
        return true;
    }

    void SendResponse(const TResponse& response) {
        if (!response.EnableKeepAlive) {
            Output().EnableKeepAlive(false);
        }
        Output() << (const THttpResponse&)response;
        Output().Finish();
        ReleaseConnection();
        INFO_LOG << response.LogPrefix << "Served '" << RequestName_ << "' -> " << (int) response.HttpCode() << "; " << ProcessingStart_.Get() << Endl;
    }
private:
    TServer& Server_;
    TProfileTimer ProcessingStart_;
    TString RequestName_;
    TRequest Request_;
};

void TRequest::Init(const TParsedHttpFull& parsed, TConnection* conn) {
    Method_ = parsed.Method;
    Path_ = parsed.Path;
    Cgi_ = parsed.Cgi;
    Query_ = TCgiParameters(parsed.Cgi);
    Connection_ = conn;
}

TString TRequest::Method() const {
    return Method_;
}

TString TRequest::Path() const {
    return Path_;
}

TString TRequest::Cgi() const {
    return Cgi_;
}

const TCgiParameters& TRequest::Query() const {
     return Query_;
}

const THttpHeaders& TRequest::Headers() const {
    return Connection_->Input().Headers();
}

IInputStream& TRequest::Body() const {
    return Connection_->Input();
}

bool TRequest::IsLocal() const {
    return Connection_->IsLocal();
}

TResponse::TResponse(const ::THttpResponse& p)
    : ::THttpResponse(p)
{
}

TResponse TResponse::CreateJson(const NProtoBuf::Message& message, HttpCodes code, bool format) {
    try {
        TString body = NProtobufJson::Proto2Json(
                message,
                NProtobufJson::TProto2JsonConfig()
                    .SetFormatOutput(format)
                    .SetMissingRepeatedKeyMode(NProtobufJson::TProto2JsonConfig::MissingKeyDefault)
                    .SetEnumMode(NProtobufJson::TProto2JsonConfig::EnumName)
                    .SetMapAsObject(true)
                    );
        return THttpResponse(code)
            .SetContent(body, "application/json");
    }
    catch (...) {
        return THttpResponse(HTTP_INTERNAL_SERVER_ERROR).SetContent(CurrentExceptionMessage());
    }
}

TResponse TResponse::CreateJson(const TString& message, HttpCodes code) {
    return THttpResponse(code).SetContent(message, "application/json");
}

TResponse TResponse::CreateText(const TString& message, HttpCodes code) {
    return THttpResponse(code).SetContent(message, "text/plain");
}

TResponse TResponse::CreateHTML(const TString& message, HttpCodes code) {
    return THttpResponse(code).SetContent(message, "text/html");
}

THandlerOptions ExternalWithoutTvm() {
    return THandlerOptions().SetUseTvm(false);
}

THandlerOptions ExternalWithTvm() {
    return THandlerOptions().SetUseTvm(true);
}

THandlerOptions Local() {
    return THandlerOptions().SetLocalOnly(true).SetUseTvm(false);
}

TServer::TCounters::TCounters(TServer & server)
    : Server(server)
{
}

void TServer::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    NMonitor::TCounter NConnections = Server.GetConnectionCount();
    ct->insert(MAKE_COUNTER_PAIR(NConnections));
    ct->insert(MAKE_COUNTER_PAIR(NHttpNotFound));
    ct->insert(MAKE_COUNTER_PAIR(NMaxConnections));
    ct->insert(MAKE_COUNTER_PAIR(NServerExceptions));

    ct->insert(MAKE_COUNTER_PAIR(NTvmNoTicket));
    ct->insert(MAKE_COUNTER_PAIR(NTvmTicketRefused));
}

bool TServer::TCallbackHolder::AcceptsRequest(const TRequest& request) const {
    if (Options.Method != request.Method()) {
        return false;
    }
    TStringBuf path = request.Path();
    path = StripStringRight(path, EqualsStripAdapter('/'));
    return AsciiEqualsIgnoreCase(Path, path);
}

void TServer::TCallbackHolder::HandleRequest(const TRequest& request, const TOnResponse& responseCb) {
    if (Options.LocalOnly && !request.IsLocal()) {
        responseCb(THttpResponse(HTTP_FORBIDDEN));
        return;
    }
    if (Options.UseTvm) {
        if (TvmService && TvmService->IsEnabled()) {
            bool ok = false;
            auto ticketPtr = request.Headers().FindHeader("X-Ya-Service-Ticket");
            if (!ticketPtr) {
                WARNING_LOG << "HTTP: Request without tvm service ticket (X-Ya-Service-Ticket) (path: " << request.Path() << ")" << Endl;
                ServerCounters.NTvmNoTicket.Inc();
            } else if (!TvmService->IsAllowedServiceTicket(ticketPtr->Value())) {
                WARNING_LOG << "HTTP: Tvm ticket refused for request (path: " << request.Path() << ")" << Endl;
                ServerCounters.NTvmTicketRefused.Inc();
            } else {
                ok = true;
            }
            if (!ok) {
                responseCb(THttpResponse(HTTP_FORBIDDEN));
                return;
            }
        }
    }
    return Cb(request, responseCb);
}

TServer::TServer()
    : Counters_(*this)
    , TvmService_(nullptr)
{
}

TServer::~TServer()
{
    Stop();
}

void TServer::Start(const NTravelProto::NAppConfig::TConfigHttp& pbConfig) {
    if (IsStarted_)
        throw yexception() << "Server is already started";

    THttpServerOptions options;
    options.SetPort(pbConfig.GetPort());
    options.SetThreads(pbConfig.GetThreads());
    options.AddBindAddress(pbConfig.GetHost());
    options.SetMaxConnections(pbConfig.GetMaxConn());
    options.SetClientTimeout(TDuration::MilliSeconds(pbConfig.GetClientTimeoutMSec()));
    options.EnableCompression(pbConfig.GetEnableCompression());

    Server_.Reset(new THttpServer(this, options));
    if (!Server_->Start()) {
        throw yexception() << "Cannot start HTTP Server at port " << options.Port << ": " << Server_->GetError();
    }
    IsStarted_.Set();
    INFO_LOG << "HTTP Server listening at " << options.Port << Endl;
}

void TServer::Stop() {
    if (!IsStarted_)
        return;

    Server_->Stop();
    Server_.Reset();
    IsStarted_.Clear();
}

void TServer::Shutdown() {
    if (Server_) {
        Server_->Shutdown();
    }
}

void TServer::RegisterCounters(NMonitor::TCounterSource& source) const {
    source.RegisterSource(&Counters_, "Http");
}

void TServer::SetTvm(const NTravel::NTvm::TTvmService* tvmService) {
    TvmService_ = tvmService;
}

void TServer::AddHandler(const TString& path, const THandlerOptions& options, TUrlCallback cb) {
    TCallbackHolder h {
        TvmService_,
        path,
        options,
        cb,
        Counters_
    };
    Callbacks_.push_back(h);
}

i64 TServer::GetConnectionCount() const {
    if (Server_) {
        return Server_->GetClientCount();
    } else {
        return 0;
    }
}

// ICallback impl

TClientRequest* TServer::CreateClient() {
    return new TConnection(*this);
}

void TServer::OnException() {
    // this method is called inside catch, so CurrentExceptionMessage is available
    ERROR_LOG << "Error occured, message: " << CurrentExceptionMessage() << Endl;
    Counters_.NServerExceptions.Inc();
}

void TServer::OnMaxConn() {
    ERROR_LOG << "Maximum number of connections reached" << Endl;
    Counters_.NMaxConnections.Inc();
}

} // namespace NHttp
} // namespace NTravel
