#pragma once

#include <travel/hotels/proto/app_config/http_server.pb.h>
#include <travel/hotels/lib/cpp/mon/counter.h>
#include <travel/hotels/lib/cpp/util/flag.h>
#include <travel/hotels/lib/cpp/tvm/tvm_service.h>

#include <library/cpp/http/server/http.h>
#include <library/cpp/http/server/options.h>
#include <library/cpp/http/server/response.h>
#include <library/cpp/http/misc/httpcodes.h>
#include <library/cpp/http/misc/parsed_request.h>
#include <library/cpp/http/io/headers.h>

#include <library/cpp/protobuf/json/proto2json.h>
#include <library/cpp/protobuf/json/json2proto.h>

#include <library/cpp/cgiparam/cgiparam.h>

namespace NTravel {
namespace NHttp {

class TConnection;

class TRequest {
public:
    void Init(const TParsedHttpFull& parsed, TConnection* conn);
    TString Method() const;
    TString Path() const;
    TString Cgi() const;
    const TCgiParameters& Query() const;
    const THttpHeaders& Headers() const;
    IInputStream& Body() const;
    bool IsLocal() const;
private:
    TString Method_;
    TString Path_;
    TString Cgi_;
    TCgiParameters Query_;
    TConnection* Connection_;
};

struct TResponse : public ::THttpResponse {
    TResponse() = default;
    TResponse(const ::THttpResponse& p);
    static TResponse CreateJson(const NProtoBuf::Message& message, HttpCodes code = HTTP_OK, bool format = false);
    static TResponse CreateJson(const TString& message, HttpCodes code = HTTP_OK);
    static TResponse CreateText(const TString& message, HttpCodes code = HTTP_OK);
    static TResponse CreateHTML(const TString& message, HttpCodes code = HTTP_OK);

    TString LogPrefix;
    bool EnableKeepAlive = true;
};

struct THandlerOptions {
    using TSelf = THandlerOptions;
#define HTTP_FLUENT_FIELD(_T_, _N_, _DEF_) \
    _T_ _N_ = _DEF_;                 \
    TSelf& Set##_N_(const _T_& v) {   \
        _N_ = v;                      \
        return *this;                 \
    }

    HTTP_FLUENT_FIELD(bool, LocalOnly, false)
    HTTP_FLUENT_FIELD(bool, UseTvm, true)
    HTTP_FLUENT_FIELD(TString, Method, "GET")
#undef HTTP_FLUENT_FIELD
};

THandlerOptions ExternalWithoutTvm();
THandlerOptions ExternalWithTvm();
THandlerOptions Local();

using TOnResponse = std::function<void (const TResponse&)>;
using TUrlCallback = std::function<void(const TRequest& request, const TOnResponse& responseCb)>;

class TServer : public THttpServer::ICallBack {
public:
    TServer();
    ~TServer();

    void Start(const NTravelProto::NAppConfig::TConfigHttp& pbConfig);
    void Stop();

    void Shutdown();// Просьба мягкго завершения

    void RegisterCounters(NMonitor::TCounterSource& source) const;

    void SetTvm(const NTravel::NTvm::TTvmService* tvmService);
    void AddHandler(const TString& path, const THandlerOptions& options, TUrlCallback cb);

    template <class TOwner>
    void AddHandler(const TString& path, const THandlerOptions& options, TOwner* owner, void (TOwner::*membFunc)(const TRequest& request, const TOnResponse& responseCb)) {
        AddHandler(path, options, [owner, membFunc](const TRequest& request, const TOnResponse& responseCb) {
            (owner->*membFunc)(request, responseCb);
        });
    }

    template <class TOwner>
    void AddHandler(const TString& path, const THandlerOptions& options, const TOwner* owner, void (TOwner::*membFunc)(const TRequest& request, const TOnResponse& responseCb) const) {
        AddHandler(path, options, [owner, membFunc](const TRequest& request, const TOnResponse& responseCb) {
            (owner->*membFunc)(request, responseCb);
        });
    }
private:
    friend class TConnection;

    struct TCounters : public NMonitor::TCounterSource {
        TServer& Server;

        NMonitor::TDerivCounter NHttpNotFound;
        NMonitor::TDerivCounter NMaxConnections;
        NMonitor::TDerivCounter NServerExceptions;

        NMonitor::TDerivCounter NTvmNoTicket;
        NMonitor::TDerivCounter NTvmTicketRefused;

        TCounters(TServer& server);

        void QueryCounters(NMonitor::TCounterTable* ct) const override;
    };

    struct TCallbackHolder {
        const NTravel::NTvm::TTvmService* TvmService;
        TString      Path;
        THandlerOptions Options;
        TUrlCallback Cb;
        TCounters& ServerCounters;

        bool AcceptsRequest(const TRequest& request) const ;
        void HandleRequest(const TRequest& request, const TOnResponse& responseCb);
    };

    TAtomicFlag IsStarted_;
    THolder<THttpServer> Server_;
    TVector<TCallbackHolder> Callbacks_;
    TCounters Counters_;
    const NTravel::NTvm::TTvmService* TvmService_;
private:
    i64 GetConnectionCount() const;

    // ICallback impl
    TClientRequest* CreateClient() override final;
    void OnException() override final;
    void OnMaxConn() override final;
};

} // namespace NHttp
} // namespace NTravel
