#include "label.h"

#include <travel/hotels/lib/cpp/util/base64.h>
#include <travel/hotels/lib/cpp/util/compress.h>

#include <openssl/sha.h>

#include <util/generic/string.h>
#include <util/generic/vector.h>
#include <util/generic/yexception.h>

namespace NTravel {
namespace NLabel {

static const TString g_EncodedPrefix = "p";

TString TLabelCodec::Encode(const TString& plainData, const TOpts& opts) const {
    TString result = plainData;
    if (opts.WithCheckSum_) {
        char sum = 0;
        for (auto c: plainData) {
            sum += c;
        }
        result += TString(sum);
    }
    result = Base64EncodeUrlShort(result);
    if (opts.WithPrefix_) {
        result = g_EncodedPrefix + result;
    }
    return result;
}

TString TLabelCodec::Encode(const ::google::protobuf::MessageLite& label, const TOpts& opts) const {
    return Encode(label.SerializeAsString(), opts);
}

TString TLabelCodec::Decode(const TString& encodedData, const TOpts& opts) const {
    TString label = encodedData;
    if (opts.WithPrefix_) {
        if (!HasPrefix(label)) {
            throw yexception() << "Invalid data prefix";
        }
        label = label.substr(g_EncodedPrefix.size());
    }
    label = Base64DecodeAny(label);
    if (label.empty()) {
        throw yexception() << "No data was decoded";
    }
    if (opts.WithCheckSum_) {
        char sum = label.back();
        label.pop_back();
        char actSum = 0;
        for (auto c: (const TString&)label) {
            actSum += c;
        }
        if (sum != actSum) {
            throw yexception() << "Checksum mismatch";
        }
    }
    return label;
}

void TLabelCodec::DecodeToMessage(const TString& encodedData, ::google::protobuf::MessageLite* message, const TOpts& opts) const {
    TString decoded = Decode(encodedData, opts);
    if (!message->ParseFromString(decoded)) {
        throw yexception() << "Failed to parse protobuf in label";
    }
}

bool TLabelCodec::HasPrefix(const TString& encodedData) const {
    return encodedData.StartsWith(g_EncodedPrefix);
}

TString CalcSHA224(const TString& str) {
    unsigned char hash[SHA224_DIGEST_LENGTH];
    SHA256_CTX ctx;
    SHA224_Init(&ctx);
    SHA224_Update(&ctx, str.data(), str.length());
    SHA224_Final(hash, &ctx);
    return TString((char*)hash, sizeof(hash));
}

TString CalcLabelHash(const ::google::protobuf::MessageLite& label) {
    TString h = CalcSHA224(label.SerializeAsString());
    // s добавляется как признак SHA224, если захотим сменить hash algo, то у него будет другой префикс и они не пересекутся
    return "s" + Base64EncodeUrlShort(h);
}

} //NLabel
} //NTravel
