#pragma once

#include "tools.h"

#include "travel/hotels/lib/cpp/util/flag.h"
#include "travel/hotels/lib/cpp/util/sizes.h"

#include "travel/hotels/proto/app_config/logger.pb.h"

#include <library/cpp/protobuf/json/proto2json.h>
#include <google/protobuf/message.h>

#include <util/thread/lfqueue.h>
#include <util/thread/factory.h>
#include <library/cpp/deprecated/atomic/atomic.h>
#include <util/system/event.h>
#include <util/string/cast.h>
#include <util/stream/file.h>
#include <util/stream/buffered.h>
#include <util/system/file.h>
#include <util/generic/variant.h>


namespace NTravel {

template <class TRecord>
class TJsonLogger : public IThreadFactory::IThreadAble {
public:
    TJsonLogger(const TString& name, NTravelProto::NAppConfig::TConfigLogger::TJsonLogger config)
        : Config_(std::move(config))
        , Name_(name)
        , Enabled_(Config_.GetEnabled())
        , IsFlushedFlag_(true)
    {
    }

    ~TJsonLogger() {
        Stop();
    }

    void Start() {
        if (Enabled_) {
            Thread_ = SystemThreadFactory()->Run(this);
        }
    }

    void Stop() {
        if (StopFlag_.TrySet()) {
            WakeUpEvent_.Signal();
            if (Thread_) {
                Thread_->Join();
                Thread_ .Reset();
            }
        }
    }

    void RegisterCounters(NMonitor::TCounterSource& source) {
        source.RegisterSource(&Counters_, "JsonLogger-" + Name_);
    }

    void Reopen() {
        if (!Enabled_) {
            return;
        }
        while (true) {
            if (ReopenFlag_.TrySet()) {
                // Чтобы два одновременно пришедших reopen-а упорядочились
                WakeUpEvent_.Signal();
                ReopenDone_.WaitI();
                return;
            }
            Sleep(TDuration::MilliSeconds(10));
        }
    }

    void AddRecord(const TRecord& rec) {
        if (!Enabled_) {
            return;
        }
        auto sz = GetSize(rec);
        Counters_.NMessagesReceived++;
        Counters_.NBytesReceived += sz;
        Counters_.NMessagesInQueue++;
        Counters_.NBytesInQueue += sz;
        IsFlushedFlag_.Clear();
        Queue_.Enqueue(rec);
        WakeUpEvent_.Signal();
    }

    bool IsFlushed() const {
        return IsFlushedFlag_;
    }

private:
    struct TCounters: public NMonitor::TCounterSource {
        NMonitor::TCounter NMessagesInQueue;
        NMonitor::TCounter NBytesInQueue;
        NMonitor::TDerivCounter NMessagesReceived;
        NMonitor::TDerivCounter NBytesReceived;
        NMonitor::TDerivCounter NMessagesWritten;
        NMonitor::TDerivCounter NBytesWritten;

        void QueryCounters(NMonitor::TCounterTable* ct) const override {
            ct->insert(MAKE_COUNTER_PAIR(NMessagesInQueue));
            ct->insert(MAKE_COUNTER_PAIR(NBytesInQueue));
            ct->insert(MAKE_COUNTER_PAIR(NMessagesReceived));
            ct->insert(MAKE_COUNTER_PAIR(NBytesReceived));
            ct->insert(MAKE_COUNTER_PAIR(NMessagesWritten));
            ct->insert(MAKE_COUNTER_PAIR(NBytesWritten));
        }
    };

    NTravelProto::NAppConfig::TConfigLogger::TJsonLogger Config_;
    TString Name_;
    const bool Enabled_;
    TAutoPtr<IThreadFactory::IThread> Thread_;
    TAtomicFlag StopFlag_;
    TAtomicFlag ReopenFlag_;
    TAtomicFlag IsFlushedFlag_;
    TAutoEvent ReopenDone_;

    TAutoEvent WakeUpEvent_;
    TLockFreeQueue<TRecord> Queue_;

    TLoggerMessageSerializer LoggerMessageSerializer_;

    TCounters Counters_;

    size_t GetSize(const TString& data) {
        return TTotalByteSize<TString>()(data);
    }

    size_t GetSize(const google::protobuf::Message& data) {
        return data.ByteSizeLong();
    }

    TString Serialize(const TString& data) {
        return data;
    }

    TString Serialize(const google::protobuf::Message& data) {
        return LoggerMessageSerializer_.Serialize(data);
    }

    void Write(const TString& data, THolder<TFileOutput>& outFile) {
        Counters_.NMessagesWritten++;
        Counters_.NBytesWritten += data.size();
        *outFile << data << Endl;
    }

    void DoExecute() override {
        THolder<TFileOutput> outFile;
        while (!StopFlag_) {
            if (!outFile.Get() || ReopenFlag_) {
                outFile.Reset(new TFileOutput(TFile(Config_.GetFile(), OpenAlways | WrOnly | ForAppend)));
                if (ReopenFlag_) {
                    ReopenFlag_.Clear();
                    ReopenDone_.Signal();
                }
            }
            TRecord rec;
            while (Queue_.Dequeue(&rec)) {
                Counters_.NMessagesInQueue--;
                Counters_.NBytesInQueue -= GetSize(rec);
                Write(Serialize(rec), outFile);
            }
            if (!WakeUpEvent_.WaitT(TDuration::Seconds(Config_.GetFlushDelaySec()))) {
                // not Signalled
                outFile->Flush();
                if (Queue_.IsEmpty()) {
                    IsFlushedFlag_.Set();
                }
                WakeUpEvent_.WaitI();
            };
        }
        ReopenDone_.Signal();// just in case, вдруг ждали...
    }
};

}// NTravel
