#include "tools.h"

#include <travel/hotels/proto/extensions/ext.pb.h>

namespace NTravel {
    TLoggerMessageSerializer::TLoggerMessageSerializer()
        : ProtoToJsonCfg(CreateProtoToJsonCfg())
        , JsonConfig(CreateJsonCfg(ProtoToJsonCfg)) {
    }

    TString TLoggerMessageSerializer::Serialize(const google::protobuf::Message& data) {
        NJson::TJsonValue value{};
        Proto2Json(data, value, ProtoToJsonCfg);
        PatchJson(&value);

        TStringStream ss;
        NJson::WriteJson(&ss, &value, JsonConfig);
        return ss.Str();
    }

    NProtobufJson::TProto2JsonConfig TLoggerMessageSerializer::CreateProtoToJsonCfg() {
        NProtobufJson::TProto2JsonConfig protoToJsonCfg;
        protoToJsonCfg.SetFormatOutput(false);
        protoToJsonCfg.SetMapAsObject(true);
        protoToJsonCfg.SetEnumMode(NProtobufJson::TProto2JsonConfig::EnumName);
        protoToJsonCfg.SetNameGenerator([](const google::protobuf::FieldDescriptor& fd) {
            auto name = fd.name();
            if (fd.options().HasExtension(NTravelProto::LogName)) {
                name = fd.options().GetExtension(NTravelProto::LogName);
            }
            if (fd.options().HasExtension(NTravelProto::ExtractNestedFieldsInLog) && fd.options().GetExtension(NTravelProto::ExtractNestedFieldsInLog)) {
                name = "!" + name;
            }
            return name;
        });

        return protoToJsonCfg;
    }

    NJson::TJsonWriterConfig TLoggerMessageSerializer::CreateJsonCfg(const NProtobufJson::TProto2JsonConfig& protoToJsonCfg) {
        NJson::TJsonWriterConfig jsonConfig;
        jsonConfig.FormatOutput = protoToJsonCfg.FormatOutput;
        jsonConfig.SortKeys = false;
        jsonConfig.ValidateUtf8 = false;
        jsonConfig.DontEscapeStrings = false;
        jsonConfig.WriteNanAsString = protoToJsonCfg.WriteNanAsString;
        return jsonConfig;
    }

    void TLoggerMessageSerializer::PatchJson(NJson::TJsonValue* json) {
        if (json->IsMap()) {
            auto& jsonAsMap = json->GetMapSafe();
            TVector<TString> keysToPatch;
            for (const auto& [k, v]: jsonAsMap) {
                if (k.StartsWith('!')) {
                    keysToPatch.push_back(k);
                }
            }
            for (const auto& key: keysToPatch) {
                auto value = jsonAsMap.at(key);
                Y_ENSURE(value.IsMap(), "Subkey " + key.substr(1) + " marked with ExtractNestedFieldsInLog, but subkey is not a map");
                jsonAsMap.erase(key);
                for (const auto& [subKey, subValue]: value.GetMapSafe()) {
                    jsonAsMap.emplace(subKey, subValue);
                }
            }
            for (auto& [k, v]: jsonAsMap) {
                PatchJson(&v);
            }
        } else if (json->IsArray()) {
            for (auto& v: json->GetArraySafe()) {
                PatchJson(&v);
            }
        }
    }
}