#pragma once

#include "tools.h"

#include <travel/hotels/proto/app_config/logger.pb.h>
#include <travel/hotels/lib/cpp/util/flag.h>

#include <logbroker/unified_agent/client/cpp/client.h>

#include <library/cpp/protobuf/json/proto2json.h>
#include <util/thread/lfqueue.h>
#include <util/thread/factory.h>
#include <library/cpp/deprecated/atomic/atomic.h>

#include <google/protobuf/message.h>

#include <utility>
#include <travel/hotels/lib/cpp/mon/counter.h>

namespace NTravel {
    template <class TRecord>
    class TUnifiedAgentLogger: public IThreadFactory::IThreadAble {
    private:
        struct TUnifiedAgentLoggerCounters : public NMonitor::TCounterSource {
            TIntrusivePtr<NUnifiedAgent::TClientCounters> UnifiedAgentClientCounters;
            TIntrusivePtr<NUnifiedAgent::TClientSessionCounters> UnifiedAgentClientSessionCounters;

            mutable NMonitor::TCounter ActiveSessionsCount;
            mutable NMonitor::TDerivCounter ClientLogDroppedBytes;

            mutable NMonitor::TDerivCounter ReceivedMessages;
            mutable NMonitor::TDerivCounter ReceivedBytes;
            mutable NMonitor::TDerivCounter AcknowledgedMessages;
            mutable NMonitor::TDerivCounter AcknowledgedBytes;
            mutable NMonitor::TCounter InflightMessages;
            mutable NMonitor::TCounter InflightBytes;
            mutable NMonitor::TDerivCounter GrpcWriteBatchRequests;
            mutable NMonitor::TCounter GrpcInflightMessages;
            mutable NMonitor::TCounter GrpcInflightBytes;
            mutable NMonitor::TDerivCounter GrpcCalls;
            mutable NMonitor::TDerivCounter GrpcCallsInitialized;
            mutable NMonitor::TDerivCounter DroppedMessages;
            mutable NMonitor::TDerivCounter DroppedBytes;
            mutable NMonitor::TDerivCounter ErrorsCount;

            TUnifiedAgentLoggerCounters()
                : UnifiedAgentClientCounters(MakeIntrusive<NUnifiedAgent::TClientCounters>())
                , UnifiedAgentClientSessionCounters(MakeIntrusive<NUnifiedAgent::TClientSessionCounters>())
            {
            }

            void QueryCounters(NMonitor::TCounterTable* ct) const override {
                ActiveSessionsCount = UnifiedAgentClientCounters->ActiveSessionsCount.Val();
                ClientLogDroppedBytes = UnifiedAgentClientCounters->ClientLogDroppedBytes.Val();

                ReceivedMessages = UnifiedAgentClientSessionCounters->ReceivedMessages.Val();
                ReceivedBytes = UnifiedAgentClientSessionCounters->ReceivedBytes.Val();
                AcknowledgedMessages = UnifiedAgentClientSessionCounters->AcknowledgedMessages.Val();
                AcknowledgedBytes = UnifiedAgentClientSessionCounters->AcknowledgedBytes.Val();
                InflightMessages = UnifiedAgentClientSessionCounters->InflightMessages.Val();
                InflightBytes = UnifiedAgentClientSessionCounters->InflightBytes.Val();
                GrpcWriteBatchRequests = UnifiedAgentClientSessionCounters->GrpcWriteBatchRequests.Val();
                GrpcInflightMessages = UnifiedAgentClientSessionCounters->GrpcInflightMessages.Val();
                GrpcInflightBytes = UnifiedAgentClientSessionCounters->GrpcInflightBytes.Val();
                GrpcCalls = UnifiedAgentClientSessionCounters->GrpcCalls.Val();
                GrpcCallsInitialized = UnifiedAgentClientSessionCounters->GrpcCallsInitialized.Val();
                DroppedMessages = UnifiedAgentClientSessionCounters->DroppedMessages.Val();
                DroppedBytes = UnifiedAgentClientSessionCounters->DroppedBytes.Val();
                ErrorsCount = UnifiedAgentClientSessionCounters->ErrorsCount.Val();

                ct->insert(MAKE_COUNTER_PAIR(ActiveSessionsCount));
                ct->insert(MAKE_COUNTER_PAIR(ClientLogDroppedBytes));

                ct->insert(MAKE_COUNTER_PAIR(ReceivedMessages));
                ct->insert(MAKE_COUNTER_PAIR(ReceivedBytes));
                ct->insert(MAKE_COUNTER_PAIR(AcknowledgedMessages));
                ct->insert(MAKE_COUNTER_PAIR(AcknowledgedBytes));
                ct->insert(MAKE_COUNTER_PAIR(InflightMessages));
                ct->insert(MAKE_COUNTER_PAIR(InflightBytes));
                ct->insert(MAKE_COUNTER_PAIR(GrpcWriteBatchRequests));
                ct->insert(MAKE_COUNTER_PAIR(GrpcInflightMessages));
                ct->insert(MAKE_COUNTER_PAIR(GrpcInflightBytes));
                ct->insert(MAKE_COUNTER_PAIR(GrpcCalls));
                ct->insert(MAKE_COUNTER_PAIR(GrpcCallsInitialized));
                ct->insert(MAKE_COUNTER_PAIR(DroppedMessages));
                ct->insert(MAKE_COUNTER_PAIR(DroppedBytes));
                ct->insert(MAKE_COUNTER_PAIR(ErrorsCount));
            }
        };

    public:
        explicit TUnifiedAgentLogger(TString name, NTravelProto::NAppConfig::TConfigLogger::TUnifiedAgentLogger config)
            : Name(std::move(name))
            , Config_(std::move(config))
            , Counters()
        {
        }

        void RegisterCounters(NMonitor::TCounterSource& source) {
            source.RegisterSource(&Counters, "UnifiedAgentClient-" + Name);
        }

        void Start() {
            if (Config_.GetEnabled()) {
                auto clientParameters = NUnifiedAgent::TClientParameters(Config_.GetUri());
                clientParameters.SetCounters(Counters.UnifiedAgentClientCounters);
                clientParameters.SetGrpcMaxMessageSize(Config_.GetGrpcMaxMessageSizeBytes());
                clientParameters.SetMaxInflightBytes(Config_.GetGrpcMaxInflightBytes());
                auto clientPtr = NUnifiedAgent::MakeClient(clientParameters);
                auto sessionParameters = NUnifiedAgent::TSessionParameters();
                sessionParameters.SetCounters(Counters.UnifiedAgentClientSessionCounters);
                ReqAnsLogClientSession_ = clientPtr->CreateSession(sessionParameters);
                Thread_ = SystemThreadFactory()->Run(this);
            }
        }

        void Stop() {
            if (StopFlag_.TrySet()) {
                StopEvent_.Signal();
                if (Thread_) {
                    Thread_->Join();
                    Thread_.Reset();
                }
                if (ReqAnsLogClientSession_) {
                    ReqAnsLogClientSession_->Close();
                }
            }
        }

        void AddRecord(const TRecord& rec) {
            if (!StopFlag_) {
                if (ReqAnsLogClientSession_) {
                    Queue_.Enqueue(rec);
                }
            }
        }

        ~TUnifiedAgentLogger() override {
            Stop();
        }

    private:
        TString Name;
        NTravelProto::NAppConfig::TConfigLogger::TUnifiedAgentLogger Config_;
        NUnifiedAgent::TClientSessionPtr ReqAnsLogClientSession_;
        TAutoPtr<IThreadFactory::IThread> Thread_;
        TLockFreeQueue<TRecord> Queue_;
        TAtomicFlag StopFlag_;
        TAutoEvent StopEvent_;
        TUnifiedAgentLoggerCounters Counters;
        TLoggerMessageSerializer LoggerMessageSerializer_;

        TString Serialize(const TString& data) {
            return data;
        }

        TString Serialize(const google::protobuf::Message& data) {
            return LoggerMessageSerializer_.Serialize(data);
        }

        void DoExecute() override {
            while (!StopFlag_ || !Queue_.IsEmpty()) {
                TRecord rec;
                while (Queue_.Dequeue(&rec)) {
                    NUnifiedAgent::TClientMessage clientMessage;
                    clientMessage.Payload = Serialize(rec);
                    ReqAnsLogClientSession_->Send(std::move(clientMessage));
                }
                StopEvent_.WaitT(TDuration::Seconds(Config_.GetFlushDelaySec()));
            }
        }
    };

}
