#include "memusage.h"

#include <library/cpp/malloc/api/malloc.h>
#include <library/cpp/lfalloc/dbg_info/dbg_info.h>

#include <util/generic/string.h>
#include <util/stream/output.h>
#include <util/system/rusage.h>

#include <initializer_list>

namespace NTravel {

static const std::initializer_list<std::pair<const char *, NAllocDbg::ELFAllocCounter>> LFCntStats = {
    {"NMallocMmap", NAllocDbg::CT_MMAP_CNT},              // number of mmapped regions
    {"NMallocMunmap", NAllocDbg::CT_MUNMAP_CNT},          // number of munmaped regions
    {"NMallocSlowAlloc", NAllocDbg::CT_SLOW_ALLOC_CNT},   // number of slow (not LF) allocations
    {"NMallocDefragment", NAllocDbg::CT_DEGRAGMENT_CNT}   // number of memory defragmentations
};

static const std::initializer_list<std::pair<const char *, NAllocDbg::ELFAllocCounter>> LFSizeStats = {
    {"MallocUserAlloc", NAllocDbg::CT_USER_ALLOC},      // accumulated size requested by user code
    {"MallocMmapSize", NAllocDbg::CT_MMAP},             // accumulated mmapped size
    {"MallocMunmapSize", NAllocDbg::CT_MUNMAP},         // accumulated unmmapped size
    {"MallocSystemAlloc", NAllocDbg::CT_SYSTEM_ALLOC},  // accumulated allocated size for internal lfalloc needs
    {"MallocSystemFree", NAllocDbg::CT_SYSTEM_FREE},    // accumulated deallocated size for internal lfalloc needs
    {"MallocSmallAlloc", NAllocDbg::CT_SMALL_ALLOC},    // accumulated allocated size for fixed-size blocks
    {"MallocSmallFree", NAllocDbg::CT_SMALL_FREE},      // accumulated deallocated size for fixed-size blocks
    {"MallocLargeAlloc", NAllocDbg::CT_LARGE_ALLOC},    // accumulated allocated size for large blocks
    {"MallocLargeFree", NAllocDbg::CT_LARGE_FREE},      // accumulated deallocated size for large blocks
};

void TMemUsageCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    i64 allocTotalSize = 0;
    const auto allocatorName = TString(NMalloc::MallocInfo().Name);
    if (allocatorName == "lfalloc_dbg") {
        for (const auto& i: LFCntStats) {
            ct->emplace(i.first, NMonitor::TCounter(NAllocDbg::GetAllocationCounterFull(i.second)));
        }
        for (const auto& i: LFSizeStats) {
            ct->emplace(i.first, NMonitor::TDerivCounter(NAllocDbg::GetAllocationCounterFull(i.second)));
        }
        const auto largeBlocks = NAllocDbg::GetAllocationCounterFull(NAllocDbg::CT_LARGE_ALLOC) - NAllocDbg::GetAllocationCounterFull(NAllocDbg::CT_LARGE_FREE);
        const auto smallBlocks = NAllocDbg::GetAllocationCounterFull(NAllocDbg::CT_SMALL_ALLOC) - NAllocDbg::GetAllocationCounterFull(NAllocDbg::CT_SMALL_FREE);
        const auto system = NAllocDbg::GetAllocationCounterFull(NAllocDbg::CT_SYSTEM_ALLOC) - NAllocDbg::GetAllocationCounterFull(NAllocDbg::CT_SYSTEM_FREE);
        ct->emplace("MallocLargeSize", NMonitor::TCounter(largeBlocks));
        ct->emplace("MallocSmallSize", NMonitor::TCounter(smallBlocks));
        ct->emplace("MallocSystemSize", NMonitor::TCounter(system));
        allocTotalSize = largeBlocks + smallBlocks + system;
    } else if (allocatorName == "jemalloc") {
        allocTotalSize = i64(NMalloc::MallocInfo().GetParam("allocated"));
    }
    ct->emplace("MallocTotalSize", NMonitor::TCounter(allocTotalSize));

    ct->emplace("ResidentSetSize", NMonitor::TCounter(TRusage::GetCurrentRSS()));
}

}// namespace NTravel
