#include "counter.h"

#include <util/datetime/base.h>
#include <util/str_stl.h>

namespace NMonitor {

///////////////////////////////////////////////////////////////////////////////
static char* PrettyNum(i64 val, char* buf, size_t size) {
    static const char shorts[] = {' ', 'K', 'M', 'G', 'T', 'P', 'E'};
    unsigned i = 0;
    i64 major = val;
    i64 minor = 0;

    for (i = 0; i < Y_ARRAY_SIZE(shorts); ++i) {
        if (major >> 10 == 0)
            break;
        else {
            minor = major - (major >> 10 << 10);
            major = major >> 10;
        }
    }

    minor = (minor * 10) >> 10;

    if (i == 0) {
        *buf = '\0';
    } else {
        snprintf(buf, size, " (%" PRId64 ".%" PRId64 "%c)", major, minor, shorts[i]);
    }

    return buf;
}

static TString PrintCounter(const TCounterBase& counter) {
    switch (counter.Type()) {
        case ECounterType::Int: {
            const TCounter& val = static_cast<const TCounter&>(counter);
            char buf[32];
            PrettyNum(val.Val(), buf, sizeof(buf));
            return ToString(val.Val()) + TString(buf);
        }
        case ECounterType::Double: {
            const TDoubleValue& val = static_cast<const TDoubleValue&>(counter);
            return ToString(val.Val());
        }
        case ECounterType::Time: {
            const TTimeValue& val = static_cast<const TTimeValue&>(counter);
            return ToString(val.Val());
        }
    }
    return TString();
}

void AddToTotals(const TCounterTable& table, TCounterTable* totals) {
    // Добавляет значения счетчиков из таблицы к totals-ам
    for (auto ti = table.begin(); ti != table.end(); ++ti) {
        if (ti->second.Flags() & TCounterFlag::NoTotals) {
            continue;
        }
        auto totalIt = totals->find(ti->first);
        if (totalIt == totals->end()) {
            totals->insert(*ti);
        } else {
            switch (ti->second.Type()) {
                case ECounterType::Int: {
                    static_cast<TCounter&>(totalIt->second) =
                            static_cast<const TCounter&>(ti->second).Val() +
                            static_cast<const TCounter&>(totalIt->second).Val();
                    break;
                }
                case ECounterType::Double: {
                    static_cast<TDoubleValue&>(totalIt->second) =
                            static_cast<const TDoubleValue&>(ti->second).Val() +
                            static_cast<const TDoubleValue&>(totalIt->second).Val();
                    break;

                }
                case ECounterType::Time:
                    // А время не суммируется
                    break;
            }
        }
    }
}

void FinishTotals(TCounterTable* totals, size_t count) {
    if (count == 0) {
        return;
    }

    for (auto ti = totals->begin(); ti != totals->end(); ++ti) {
        // Пропускаем таблицу, если для неё не надо вычислять среднее
        if ((ti->second.Flags() & TCounterFlag::Average) == 0) {
            continue;
        }
        // Вычисление среднего для некоторых счетчиков
        switch (ti->second.Type()) {
            case ECounterType::Int: {
                static_cast<TCounter&>(ti->second) =
                    static_cast<const TCounter&>(ti->second).Val() / count;
                break;
            }
            case ECounterType::Double: {
                static_cast<TDoubleValue&>(ti->second) =
                    static_cast<const TDoubleValue&>(ti->second).Val() / count;
                break;

            }
            case ECounterType::Time:
                // А время не делится
                break;
        }
    }
}

TCounterSource::TCounterSource(const TString& label, const bool calcTotal)
    : Label_(label)
    , CalcTotal_(calcTotal)
{
}

void TCounterSource::RegisterSource(const TAbstractCounterSource* source, const TString& name) {
    TWriteGuard g(Lock_);
    Sources_.insert(std::make_pair(name, source));
}

void TCounterSource::UnregisterSource(const TString& name) {
    TWriteGuard g(Lock_);
    Sources_.erase(name);
}

void TCounterSource::QueryCounters(TCounterTable* ct) const {
    Y_UNUSED(ct);
}

void TCounterSource::QueryTables(const TCounterTableName& name, TNamedCounterTables* tables) const {
    TCounterTable totals;
    size_t sourcesSize;

    {
        TReadGuard g(Lock_);
        // Запрос всех подтаблиц
        for (auto si = Sources_.begin(); si != Sources_.end(); ++si) {
            TCounterTableName subName(name);
            subName.push_back(std::make_pair(Label_, si->first));

            // Своя таблица источника
            tables->emplace_back();
            auto& ti = tables->back();
            ti.first = subName;
            si->second->QueryCounters(&ti.second);
            if (CalcTotal_) {
                AddToTotals(ti.second, &totals);
            }
            // Все подтаблицы источника
            si->second->QueryTables(subName, tables);
        }

        sourcesSize = Sources_.size();
    }

    if (CalcTotal_) {
        FinishTotals(&totals, sourcesSize);
        tables->emplace_back();
        auto& ti = tables->back();
        ti.first = name;
        ti.first.push_back(std::make_pair(Label_, "_TOTAL_"));
        ti.second = std::move(totals);
    }
}

} // namespace NMonitor

template <>
void Out< ::NMonitor::TCounterBase>(IOutputStream& out, const ::NMonitor::TCounterBase& c) {
    out << NMonitor::PrintCounter(c);
}
