#pragma once

#include <library/cpp/charset/ci_string.h>

#include <util/datetime/base.h>
#include <util/generic/map.h>
#include <util/generic/cast.h>
#include <util/generic/string.h>
#include <util/generic/utility.h>
#include <util/generic/vector.h>
#include <util/string/vector.h>
#include <library/cpp/deprecated/atomic/atomic.h>
#include <util/system/defaults.h>
#include <util/system/rwlock.h>

#include <utility>

namespace NMonitor {

using TCounterFlags = ui8;

struct TCounterFlag {
    enum TCounterFlags : ui8 {
        Deriv    = 1, // Пометка для solomon что надо рисовать производную от счетчика
        Average  = 2, // При подсчете суммарных значений брать среднее, а не общую сумму
        NoTotals = 4, // Не помещать счетчик в суммарные значения
    };
};

enum class ECounterType {
    Int,
    Double,
    Time
};

class TCounterBase {
public:
    using TRawValue = intptr_t ;

    inline TCounterBase(ECounterType type, TRawValue value, TCounterFlags flags = 0)
        : Type_(type)
        , Flags_(flags)
        , Value_(value)
    {
    }

    TCounterBase(const TCounterBase& rhs)
        : Type_(rhs.Type_)
        , Flags_(rhs.Flags_)
        , Value_(AtomicGet(rhs.Value_))
    {
    }

    TCounterBase& operator = (const TCounterBase& rhs) {
        Type_  = rhs.Type_;
        Flags_ = rhs.Flags_;
        AtomicSet(Value_, AtomicGet(rhs.Value_));
        return *this;
    }

    inline void SetDerivative() {
        Flags_ |= TCounterFlag::Deriv;
    }

    inline TCounterFlags Flags() const {
        return Flags_;
    }

    inline ECounterType Type() const {
        return Type_;
    }

protected:
    ECounterType Type_;
    TCounterFlags Flags_;
    TAtomic Value_;
};

/**
 */
class TCounter : public TCounterBase {
public:
    using TValue = intptr_t;

    inline TCounter(const TValue value = 0, TCounterFlags flags = 0)
        : TCounterBase(ECounterType::Int, value, flags)
    { }

    inline TValue Val() const noexcept {
        return Value_;
    }

    inline operator TValue () const noexcept {
        return Value_;
    }

    inline void Inc() noexcept {
        AtomicIncrement(Value_);
    }

    inline void Dec() noexcept {
        AtomicDecrement(Value_);
    }

    // operator overloads convinient
    inline void operator ++ () noexcept {
        Inc();
    }

    inline void operator ++ (int) noexcept {
        Inc();
    }

    inline void operator -- () noexcept {
        Dec();
    }

    inline void operator -- (int) noexcept {
        Dec();
    }

    inline void operator += (const TRawValue rhs) noexcept {
        AtomicAdd(Value_, rhs);
    }

    inline void operator -= (const TRawValue rhs) noexcept {
        AtomicAdd(Value_, -rhs);
    }

    inline TCounter& operator = (const TValue rhs) noexcept {
        AtomicSet(Value_, rhs);
        return *this;
    }
};

class TDerivCounter : public TCounter {
public:
    inline TDerivCounter(const TCounter::TRawValue value = 0)
        : TCounter(value, TCounterFlag::Deriv)
    {
    }
};

class TDoubleValue : public TCounterBase {
public:
    using TValue = double;

    static_assert(sizeof(TValue) == sizeof(TRawValue), "Size of double should be same as intptr_t");

    inline TDoubleValue(const TValue value = 0.0, TCounterFlags flags = 0)
        : TCounterBase(ECounterType::Double, BitCast<TRawValue>(value), flags)
    {
    }

    inline double Val() const noexcept {
        return BitCast<double>(Value_);
    }

    inline TDoubleValue& operator = (const double value) noexcept {
        AtomicSet(Value_, BitCast<TRawValue>(value));
        return *this;
    }
};

class TDoubleAverageValue : public TDoubleValue {
public:
    inline TDoubleAverageValue(const double value = 0.0)
        : TDoubleValue(value, TCounterFlag::Average)
    {
    }
};

class TTimeValue : public TCounterBase {
public:
    using TValue = TInstant;

    static_assert(sizeof(TInstant::TValue) == sizeof(TRawValue), "Size of time should be same as intptr_t");

    inline TTimeValue(const TInstant& value = TInstant())
        : TCounterBase(ECounterType::Time, value.MicroSeconds(), TCounterFlag::NoTotals)
    {
    }

    inline TInstant Val() const noexcept {
        return TInstant::MicroSeconds(Value_);
    }

    inline TTimeValue& operator = (const TInstant& value) noexcept {
        AtomicSet(Value_, value.MicroSeconds());
        return *this;
    }
};

#define MAKE_NAMED_COUNTER_PAIR(name, var) std::make_pair(name, var)
#define MAKE_COUNTER_PAIR(var) MAKE_NAMED_COUNTER_PAIR(#var, var)

using TCounterTable = TMap<TString, TCounterBase>;

// Элемент имени таблицы - это пара "метка Solomon, значение метки"
using TCounterTableNameElement = std::pair<TString, TCiString>;
// Имя таблицы - произвольная последовательность элементов имени
using TCounterTableName = TVector<TCounterTableNameElement>;

// Список таблиц с именами
using TNamedCounterTables = TVector<std::pair<TCounterTableName, TCounterTable>>;

void AddToTotals(const TCounterTable& table, TCounterTable* totals);
void FinishTotals(TCounterTable* totals, size_t count);

/**
 * Интерфейс для хранилища счётчиков.
 */
class TAbstractCounterSource {
public:
    //! Получить текущее значение счётчиков.
    virtual void QueryCounters(TCounterTable* ct) const = 0;

    //! Получить значения своих счетчиков и всех подтаблиц.
    //! Name - имя текущей таблицы, к подтаблицам дописываются их имена
    virtual void QueryTables(const TCounterTableName& name, TNamedCounterTables* tables) const = 0;
};

/**
 * Класс, у которого можно запросить значение счётчиков и подтаблиц.
 */
class TCounterSource : public TAbstractCounterSource {
public:
    explicit TCounterSource(const TString& label = TString(), const bool calcTotal = false);

    virtual ~TCounterSource()
    { }

    virtual void QueryCounters(TCounterTable* ct) const override;

    virtual void QueryTables(const TCounterTableName& name, TNamedCounterTables* tables) const override;

    //! Зарегистрировать подисточник
    void RegisterSource(const TAbstractCounterSource* source, const TString& name);

    //! Разрегистрировать подисточник
    void UnregisterSource(const TString& name);

private:
    using TSourceMap = TMap<TString, const TAbstractCounterSource*>;

    //! Имя метки для solomon
    const TString Label_;
    //! Нужно ли вести подсчет общих значений
    const bool   CalcTotal_;
    TRWMutex     Lock_;
    TSourceMap   Sources_;
};

class TCounterSourceWithTotals : public TCounterSource {
public:
    TCounterSourceWithTotals(const TString& label = TString())
        : TCounterSource(label, true)
    {
    }
};

} // namespace NMonitor
