#include "page.h"

#include <library/cpp/monlib/deprecated/json/writer.h>

#include <util/stream/output.h>
#include <util/string/cast.h>
#include <util/string/split.h>

namespace NMonitor {

///////////////////////////////////////////////////////////////////////////////

TBasicPage::TBasicPage()
    : TCounterSource("subgroup")
    , Started_(Now())
    , TableCountHint_(0)
{
}

TBasicPage::~TBasicPage()
{ }

void TBasicPage::AddToHttpService(THttpService& service) {
    // Для запроса всех счетчиков путь "/c"
    // Для запроса подсчетчиков - путь "/c/{шаблон1}/{шаблон2}...
    // В этом случае выводятся только те таблицы, компоненты имени которых начинаются
    // с указанных шаблонов (причем, регистроНЕзависимо).
    // Пример: Есть таблица "Zora" / "rotor". Чтобы её запросить - делаем запрос пути /c/zo/r

    service.AddPage("c", "Plaintext counters",
                    [this](const TStringBuf& path, IOutputStream& os){ Print(path, os); } );
    service.AddPage("json", "JSON counters",
                    [this](const TStringBuf&, IOutputStream& os){ PrintJson(os); } );
}

TNamedCounterTables TBasicPage::FetchTables() const {
    TNamedCounterTables tables;
    tables.reserve(AtomicGet(TableCountHint_));
    QueryTables(TCounterTableName(), &tables);
    AtomicSet(TableCountHint_, tables.size());
    return tables;
}

void TBasicPage::Print(IOutputStream& out) const {
    Print(TStringBuf(""), out);
}

void TBasicPage::Print(const TStringBuf& path, IOutputStream& out) const {
    TVector<TStringBuf> selectors;
    Split(path, "/", selectors);

    const TNamedCounterTables tables(FetchTables());

    //
    // Output values
    //

    out << "Started at " << Started_ << '\n';

    for (auto ti = tables.begin(); ti != tables.end(); ++ti) {
        if (ti->second.empty()) {
            continue;
        }
        if (!CheckTableName(selectors, ti->first)) {
            continue;
        }

        out << "=== ";
        bool first = true;
        for (auto ni = ti->first.begin(); ni != ti->first.end(); ++ni) {
            if (first) {
                first = false;
            } else {
                out << " / ";
            }
            out << ni->second;
        }
        out << " ===\n";

        for (auto ci = ti->second.begin(); ci != ti->second.end(); ++ci) {
            out << ci->first << ": \t" << ci->second << '\n';
        }
    }
}

bool TBasicPage::CheckTableName(const TVector<TStringBuf>& selectors, const TCounterTableName& tableName) const {
    auto si = selectors.begin();
    for (auto ni = tableName.begin(); ni != tableName.end() && si != selectors.end(); ++ni, ++si) {
        if (!ni->second.StartsWith(*si)) {
            return false;
        }
    }
    return si == selectors.end();
}

class TBasicPage::TNamedTableJsonWriter {
public:
    TNamedTableJsonWriter(IOutputStream& os)
        : Writer(&os)
    {
    }

    void WriteTables(const TNamedCounterTables& tables) {
        Writer.OpenDocument();
        {
            Writer.OpenMetrics();
            {
                for (auto ti = tables.begin(); ti != tables.end(); ++ti) {
                    for (auto ci = ti->second.begin(); ci != ti->second.end(); ++ci) {
                        WriteSensor(ti->first, ci->first, ci->second);
                    }
                }
            }
            Writer.CloseMetrics();
        }
        Writer.CloseDocument();
    }

private:
    void WriteSensor(const TCounterTableName& tableName, const TString& sensorName, const TCounterBase& counter) {
        if (counter.Type() == ECounterType::Time)
            return;
        Writer.OpenMetric();
        {

            Writer.OpenLabels();
            {
                for (auto ni = tableName.begin(); ni != tableName.end(); ++ni) {
                    Writer.WriteLabel(ni->first, ni->second);
                }
                Writer.WriteLabel(TStringBuf("sensor"), sensorName);
            }
            Writer.CloseLabels();
            if (counter.Flags() & TCounterFlag::Deriv) {
                Writer.WriteModeDeriv();
            }
            switch (counter.Type()) {
                case ECounterType::Int:
                    Writer.WriteValue(static_cast<const TCounter&>(counter).Val());
                    break;
                case ECounterType::Double:
                    Writer.WriteDoubleValue(static_cast<const TDoubleValue&>(counter).Val());
                    break;
                case ECounterType::Time:
                    break;
            }
        }
        Writer.CloseMetric();
    }

private:
    NMonitoring::TDeprecatedJsonWriter Writer;
};

void TBasicPage::PrintJson(IOutputStream& os) const {
    TNamedTableJsonWriter(os).WriteTables(FetchTables());
}

///////////////////////////////////////////////////////////////////////////////

} // namespace NMonitor
