#include "service.h"

#include <library/cpp/http/server/http.h>
#include <library/cpp/http/misc/parsed_request.h>

#include <util/generic/string.h>
#include <util/generic/hash.h>
#include <util/system/rwlock.h>

namespace NMonitor {

///////////////////////////////////////////////////////////////////////////////

class THttpService::TImpl : public THttpServer::ICallBack {

    class TRequest : public TRequestReplier {
    public:
        TRequest(const THttpService::TImpl* service)
            : Service_(service)
        {
        }

        bool DoReply(const TReplyParams& p) override {
            try {
                TParsedHttpFull parsed(p.Input.FirstLine());
                Service_->Reply(parsed.Path, p.Output);
            } catch (...) {
                p.Output << "HTTP/1.1 500 Internal Server Error\r\n\r\n"
                         << CurrentExceptionMessage();
            }
            return true;
        }

    private:
        const THttpService::TImpl* Service_;
    };

public:
    TImpl() {
        // Справочная стартовая страница
        AddPage("", "",
                [this](const TStringBuf&, IOutputStream& out) {
                    for (auto pi = Pages_.begin(); pi != Pages_.end(); ++pi) {
                        if (pi->first) {
                            out << "<a href=\"/" << pi->first << "\">"
                                << "/" << pi->first << " - " << pi->second.Descr
                                << "</a><br>";
                        }
                    }
                }
        );
    }

    ~TImpl() {
        Stop();
    }

    void AddPage(const TString& path, const TString& descr, THttpPage cb) {
        TWriteGuard g(Lock_);
        TPageInfo info;
        info.Descr = descr;
        info.Page = cb;
        Pages_.insert(std::make_pair(path, info));
    }

    void Reply(TStringBuf path, IOutputStream& output) const {
        // Путь всегда начинается с /, и мы его отрезаем
        if (path.StartsWith('/')) {
            path.Skip(1);
        }
        // принимаем во внимание только первый компонент пути,
        // остальные части отдаем странице-обработчику
        TStringBuf pathStart, pathRest;
        path.Split('/', pathStart, pathRest);

        {
            TReadGuard g(Lock_);
            auto pi = Pages_.find(pathStart);
            if (pi == Pages_.end()) {
                output << "HTTP/1.1 404 Not found\r\n\r\n";
            } else {
                output << "HTTP/1.1 200 Ok\r\n\r\n";
                pi->second.Page(pathRest, output);
            }
        }

        output.Flush();
    }

    void Start(int port) {
        Http_.Reset(
            new THttpServer(this, THttpServer::TOptions(port))
        );
        if (!Http_->Start()) {
            ythrow yexception() << "Cannot start Monitoring Server at port " << port << ": " << Http_->GetError();
        }
    }

    void Stop() {
        Http_.Destroy();
    }

private:
    // THttpServer::ICallBack
    TClientRequest* CreateClient() final {
        return new TRequest(this);
    }

private:
    struct TPageInfo {
        THttpPage Page;
        TString Descr;
    };

    THolder<THttpServer> Http_;

    TRWMutex Lock_;
    THashMap<TString, TPageInfo> Pages_;
};

THttpService::THttpService()
    : Impl_(new TImpl())
{
}

THttpService::~THttpService() {
}

void THttpService::AddPage(const TString& path, const TString& descr, THttpPage cb) {
    Impl_->AddPage(path, descr, cb);
}

void THttpService::Start(int port) {
    Impl_->Start(port);
}

void THttpService::Stop() {
    Impl_->Stop();
}

} // namespace NMonitor
