#pragma once

#include "invalidation_rules.h"

#include <travel/hotels/lib/cpp/data/data.h>
#include <travel/hotels/lib/cpp/mon/counter.h>
#include <travel/hotels/lib/cpp/util/flag.h>

#include <travel/hotels/proto/app_config/cache_invalidator.pb.h>
#include <travel/hotels/proto/offer_invalidation/event.pb.h>
#include <travel/hotels/proto/offerbus_messages/bus_messages.pb.h>

#include <library/cpp/containers/concurrent_hash/concurrent_hash.h>

#include <util/generic/hash.h>
#include <util/system/event.h>
#include <util/thread/factory.h>

namespace NTravel {
    class TCacheInvalidationService {
    public:
        explicit TCacheInvalidationService(const NTravelProto::NAppConfig::TConfigCacheInvalidationService& config);
        void RegisterCounters(NMonitor::TCounterSource& counters);
        void Start();
        void Stop();
        bool IsReady() const;

        bool ProcessCacheInvalidationMessage(const NTravelProto::NOfferBus::TOfferInvalidationMessage& cacheInvalidationMessage);
        bool IsInvalidated(const TPreKey& preKey, NOrdinalDate::TOrdinalDate checkIn, NOrdinalDate::TOrdinalDate checkOut, TInstant timestamp) const;
        TInstant GetInvalidationTimestamp(const TPreKey& preKey, NOrdinalDate::TOrdinalDate checkIn, NOrdinalDate::TOrdinalDate checkOut) const;

    private:
        struct TCounters: public NMonitor::TCounterSource {
            NMonitor::TDerivCounter NInvalidInvalidationRules;
            NMonitor::TDerivCounter NInvalidationRulesWithUnknownSource;
            NMonitor::TDerivCounter NInvalidationRulesWithBannedSource;
            NMonitor::TCounter NDataBytes;
            NMonitor::TCounter NInvalidationRules;
            NMonitor::TCounter NHotelsWithRulesSizeSoftLimitViolation;
            NMonitor::TCounter NHotelsWithRulesSizeHardLimitViolation;
            NMonitor::TCounter MemorySoftLimitViolation;
            NMonitor::TCounter MemoryHardLimitViolation;

            void QueryCounters(NMonitor::TCounterTable* ct) const override;
        };

        class TInvalidationInfo {
        public:
            bool AddRule(TCacheInvalidationRuleByTargetDates rule);
            bool AddRule(TCacheInvalidationRuleByCheckInOut rule);
            TInstant GetInvalidationTimestamp(NOrdinalDate::TOrdinalDate checkIn, NOrdinalDate::TOrdinalDate checkOut) const;
            size_t GetRulesCount() const;
            size_t GetAllocSize() const;
            size_t CalcTotalByteSize() const;
            THashMap<std::pair<NOrdinalDate::TOrdinalDate, NOrdinalDate::TOrdinalDate>, TCacheInvalidationRuleByTargetDates> RulesByTargetDates;
            THashMap<std::pair<TMaybe<NOrdinalDate::TOrdinalDate>, TMaybe<NOrdinalDate::TOrdinalDate>>, TCacheInvalidationRuleByCheckInOut> RulesByCheckInCheckOut;
        };

        void RunCleanup();
        bool IsAliveRule(const TCacheInvalidationRuleBase* rule, TInstant now) const;

        constexpr static size_t NUM_BUCKETS = 64;

        const bool Enabled;
        const TDuration CleanupDelay;
        const TDuration InvalidationEventLifetime;
        const bool DropRecordsWithUnknownInvalidationSource;
        THashSet<NTravelProto::NOfferInvalidation::EOfferInvalidationSource> BannedInvalidationSources;
        size_t RulesSizeSoftLimit;
        size_t RulesSizeHardLimit;
        size_t MemorySoftLimitBytes;
        size_t MemoryHardLimitBytes;

        TCounters Counters;

        TConcurrentHashMap<TPreKey, TInvalidationInfo, NUM_BUCKETS, TRWMutex> Invalidations;

        TAtomicFlag Started;
        TAtomicFlag Stopping;
        TAutoEvent StopEvent;

        THolder<IThreadFactory::IThread> CleanupThread;
    };
}
