#include "ordinal_date.h"

#include <util/datetime/parser.h>
#include <util/string/printf.h>
#include <util/string/cast.h>

namespace NTravel {

namespace NOrdinalDate {

TOrdinalDate FromString(const TString& date) {
    int y, m, d;
    try {
        if ((date.length() == 10) && (date[4] == '-') && (date[7] == '-')) {
            y = ::FromString(date.substr(0, 4));
            m = ::FromString(date.substr(5, 2));
            d = ::FromString(date.substr(8, 2));
        } else if (date.length() == 8) {
            y = ::FromString(date.substr(0, 4));
            m = ::FromString(date.substr(4, 2));
            d = ::FromString(date.substr(6, 2));
        } else {
            throw TInvalidDateException() << "Invalid date format for date '" << date << "'";
        }
    } catch (const TFromStringException& exc) {
        throw TInvalidDateException() << "Cannot parse date '" << date << "': " << CurrentExceptionMessage();
    }
    return FromParts(y, m, d);
}

TString ToString(TOrdinalDate date) {
    int y, m, d;
    ToParts(date, y, m, d);
    return Sprintf("%04d-%02d-%02d", y, m, d);
}

TString PlusDays(const TString& date, int diff) {
    return ToString(FromString(date) + diff);
}

TOrdinalDate FromInstant(const TInstant& t) {
    TOrdinalDate res = t.Days();
    if (res == g_DateZero) {
        throw TInvalidDateException() << "Invalid zero date";
    }
    return res;
}

TInstant ToInstant(TOrdinalDate date) {
    return TInstant::Days(date);
}

void ToParts(TOrdinalDate date, int& year, int& month, int& day) {
    TInstant instant = ToInstant(date);
    struct tm t;
    instant.GmTime(&t);
    year = t.tm_year + 1900;
    month = t.tm_mon + 1;
    day = t.tm_mday;
}

TOrdinalDate FromParts(int year, int month, int day) {
    TDateTimeFieldsDeprecated fields;
    Zero(fields);
    fields.Year = year;
    fields.Month = month;
    fields.Day = day;
    TInstant instant = fields.ToInstant(TInstant::MicroSeconds(0));
    if (instant.GetValue() == 0) {
        throw TInvalidDateException() << "Failed to convert '" << year << "-" << month << "-" << day << "' to TInstant";
    }
    return FromInstant(instant);
}

int GetWeekDay(TOrdinalDate date) {
    TInstant instant = ToInstant(date);
    struct tm t;
    instant.GmTime(&t);
    return t.tm_wday;
}

} // NOrdinalDate

} // NTravel
