#include "packed_ages.h"

#include <util/datetime/parser.h>
#include <util/string/printf.h>
#include <util/string/cast.h>
#include <util/generic/algorithm.h>

namespace NTravel {

namespace NPackedAges {

constexpr ui8 g_MaxAdults = 15;
constexpr ui8 g_MaxSpecKidsAge2 = 15;
constexpr ui8 g_MaxSpecKidsAge15 = 15;
constexpr ui8 g_MaxKids = 4;

constexpr ui8 g_AgeBucketCount = 17;
constexpr ui8 g_AgeBucketAdult = 16;

TPackedAges FromString(const TString& ages) {
    ui8 adultsCount = 0;
    ui8 specKidsAge2Count = 0;
    ui8 specKidsAge15Count = 0;
    ui8 kidsCount = 0;
    ui8 kidsAges[g_MaxKids] = {};
    TStringBuf agesBuf(ages);
    while (!agesBuf.empty()) {
        TStringBuf ageBuf = agesBuf.NextTok(',');
        ui8 age;
        try {
            age = ::FromString(ageBuf);
        } catch (const TFromStringException&) {
            throw TInvalidAgesException() << CurrentExceptionMessage();
        }
        if (age == 2) {
            if (specKidsAge2Count > g_MaxSpecKidsAge2) {
                throw TInvalidAgesException() << "Too much kids age 2";
            }
            ++specKidsAge2Count;
        } else if (age == 15) {
            if (specKidsAge15Count > g_MaxSpecKidsAge15) {
                throw TInvalidAgesException() << "Too much kids age 15";
            }
            ++specKidsAge15Count;
        } else if (age > 0 && age < 16) {
            // kid
            if (kidsCount >= g_MaxKids) {
                throw TInvalidAgesException() << "Too much kids";
            }
            kidsAges[kidsCount] = age;
            ++kidsCount;
        } else if (age == 88) {
            if (adultsCount >= g_MaxAdults) {
                throw TInvalidAgesException() << "Too much adults";
            }
            ++adultsCount;
        } else {
            throw TInvalidAgesException() << "Invalid age " << (int)age;
        }
    }
    Sort(std::begin(kidsAges), std::end(kidsAges)); // Для одинакового результата в случае "1,2" и "2,1"
    TPackedAges res = adultsCount | ((ui32)specKidsAge2Count << 4) | ((ui32)specKidsAge15Count << 8);
    ui8 shift = 12;
    for (ui8 k = 0; k < g_MaxKids; ++k) {
        res |= ((ui32)kidsAges[k]) << shift;
        shift += 4;
    }
    if (res == g_AgesZero) {
        throw TInvalidAgesException() << "Invalid zero ages: no adults, no kids";
    }
    return res;
}

TString ToString(TPackedAges packedAges) {
    ui8 ageBuckets[g_AgeBucketCount] = {};
    ageBuckets[g_AgeBucketAdult] = packedAges & 15; //adults
    ageBuckets[2] = (packedAges >> 4) & 15;
    ageBuckets[15] = (packedAges >> 8) & 15;
    ui8 shift = 12;
    for (ui8 k = 0; k < g_MaxKids; ++k) {
        ui8 age = (packedAges >> shift) & 15;
        if (age != 0) {
            ageBuckets[age] += 1;
        }
        shift += 4;
    }
    TString res;
    for (ui8 k = 0; k < g_AgeBucketCount; ++k) {
        for (ui8 n = 0; n < ageBuckets[k]; ++n) {
            if (res) {
                res += ",";
            }
            if (k == g_AgeBucketAdult) {
                res += "88";
            } else {
                res += ::ToString(k);
            }
        }
    }
    return res;
}

} // NPackedAges

} // NTravel
