#include "permalink_to_original_ids_mapper.h"

#include <util/generic/ptr.h>
#include <util/system/rwlock.h>
#include <util/generic/algorithm.h>

#include <type_traits>

namespace NTravel::NPermalinkMappers {

bool TPermalinkToOriginalIdsMappingRec::Contains(const THotelId& hId) const {
    auto it = LowerBound(std::cbegin(PartnerIds), std::cend(PartnerIds), hId);
    return (it != std::cend(PartnerIds)) && !(hId < *it);
}

size_t TPermalinkToOriginalIdsMappingRec::GetAllocSize() const {
    size_t allocSize = 0;
    for (const auto& partnerId: PartnerIds) {
        allocSize += sizeof partnerId + partnerId.GetAllocSize();
    }
    return allocSize;
}

void TPermalinkToOriginalIdsMapper::SetPartnerIdByCode(const THashMap<TString, EPartnerId>& partnerIdByCode) {
    TWriteGuard g(PartnerIdByCodeLock_);
    PartnerIdByCode_ = partnerIdByCode;
    // Вообще полагается перечитать таблицу из YT, вдруг соответствие поменялось. Но это не реализовано. TODO
}

size_t TPermalinkToOriginalIdsMapper::GetAllocSize(const TMappingElement& element) const {
    return element.second.GetAllocSize();
}

void TPermalinkToOriginalIdsMapper::OnConvert(const NYT::TNode& row, NTravelProto::NPermalinkMappers::TPermalinkToOriginalIdsMapperRec* proto) const {
    proto->SetPermalink(ExtractPermalinkFromYtRow(row));
    const auto& rowMap = row.AsMap();
    {
        TReadGuard g(PartnerIdByCodeLock_);
        for (const auto& [code, pId]: PartnerIdByCode_) {
            auto rIt = rowMap.find(code);
            if (rIt != rowMap.end()) {
                const auto& colNode = rIt->second;
                if (colNode.IsNull()) {
                    continue;
                }
                auto mappingForPartner = proto->AddMapping();
                mappingForPartner->SetPartnerId(pId);
                for (const auto& origIdNode: colNode.AsList()) {
                    mappingForPartner->AddOriginalId(origIdNode.AsString());
                }
            }
        }
    }
    // Вариант для не транспонированной таблицы
    EPartnerId partnerId = ExtractPartnerIdFromYtRow(row);
    TString originalId = ExtractOriginalIdFromYtRow(row);
    if ((partnerId != EPartnerId::PI_UNUSED) && originalId) {
        auto mappingForPartner = proto->AddMapping();
        mappingForPartner->SetPartnerId(partnerId);
        mappingForPartner->AddOriginalId(originalId);
    }
}

void TPermalinkToOriginalIdsMapper::OnData(const NTravelProto::NPermalinkMappers::TPermalinkToOriginalIdsMapperRec& proto) {
    NewMapping_->push_back({TPermalink(proto.GetPermalink()), {}});
    auto& partnerIds = NewMapping_->back().second.PartnerIds;
    for (const auto& p: proto.GetMapping()) {
        auto partnerId = p.GetPartnerId();
        for (const auto& oid: p.GetOriginalId()) {
            partnerIds.push_back({partnerId, oid});
        }
    }
    partnerIds.shrink_to_fit();
    Sort(partnerIds);
}

void TPermalinkToOriginalIdsMappingRecJoiner::Join(TPermalinkToOriginalIdsMapper::TValue* to, TPermalinkToOriginalIdsMapper::TValue* from) {
    auto& dst = to->PartnerIds;
    auto& src = from->PartnerIds;
    TVector<THotelId> tmp;
    tmp.reserve(dst.size() + src.size());
    std::set_union(std::begin(dst), std::end(dst), std::begin(src), std::end(src), std::back_inserter(tmp));
    tmp.erase(Unique(std::begin(tmp), std::end(tmp)), std::cend(tmp));
    tmp.shrink_to_fit();
    tmp.swap(dst);
    src.clear();
    src.shrink_to_fit();
}

}// namespace NTravel::NPermalinkMappers
