#pragma once

#include "tools.h"

#include <library/cpp/logger/global/global.h>

#include <util/generic/string.h>
#include <util/folder/path.h>
#include <util/system/env.h>
#include <util/system/user.h>

namespace NTravel {

template <class TConfig>
bool OverrideConfigFromFile(TConfig* pbCfg, const TFsPath& file, const TString& descr) {
    if (file.Exists()) {
        INFO_LOG << "Reading service " << descr << " config from file " << file << Endl;
        TConfig pbCfgOver;
        NProtobuf::ParseTextFromFile(file, &pbCfgOver, true);
        pbCfg->MergeFrom(pbCfgOver);
        return true;
    } else {
        INFO_LOG << "Skip inexistent " << descr << " config file " << file << Endl;
        return false;
    }
}

template <class TConfig>
TConfig ReadAppConfig(const TString& appName, const TFsPath& path, const TString& env) {
    TConfig pbCfg;
    bool read = false;
    TString suffix = "-app";
    TString ext = ".config";
    if (OverrideConfigFromFile(&pbCfg, path / (appName + suffix + ext), "base")) {
        read = true;
    }
    if (OverrideConfigFromFile(&pbCfg, path / (appName + suffix + "-" + env + ext), "env")) {
        read = true;
    }
    if (OverrideConfigFromFile(&pbCfg, path / (appName + suffix + "-override" + ext), "env-override")) {  // Used by Nanny
        read = true;
    }
    if (!pbCfg.IsInitialized()) {
        throw yexception() << "Message of type \"" << pbCfg.GetDescriptor()->full_name()
                           << "\" is missing required fields: " << pbCfg.InitializationErrorString();
    }
    if (!read) {
        throw yexception() << "No config files was found at path " << path << Endl;
    }
    return pbCfg;
}

template <class TConfig>
TConfig ReadAppConfigWithSubstitutions(const TString& appName, const TFsPath& path, const TString& env, const TString& ytTokenPath, const TString& ypTokenPath) {
    TConfig pbCfg = ReadAppConfig<TConfig>(appName, path, env);
    if (ytTokenPath) {
        INFO_LOG << "YT Token Path override to " << ytTokenPath << Endl;
        NTravel::NProtobuf::SetStringFieldRecursive("YtTokenPath", ytTokenPath, "Config", &pbCfg, false);
    }

    if (ypTokenPath) {
        INFO_LOG << "YP Token Path override to " << ypTokenPath << Endl;
        NTravel::NProtobuf::SetStringFieldRecursive("YpTokenPath", ypTokenPath, "Config", &pbCfg, false);
    }
    NTravel::NProtobuf::ReplaceStringFieldValueRecursive("${USER}", GetUsername(), "Config", &pbCfg, false);
    NTravel::NProtobuf::ReplaceStringFieldValueRecursive("${DATA_PATH}", pbCfg.GetOther().GetDataPath(), "Config", &pbCfg, false);

    for (const auto& [value, inner]: NTravel::NProtobuf::FindAllConfigSubstitutions("Config", &pbCfg)) {
        if (inner.StartsWith("ENV:")) {
            auto varName = inner.substr(4);
            auto realValue = GetEnv(varName);
            Y_ENSURE(!realValue.Empty(), "Env variable '" + varName + "' is not set");
            NTravel::NProtobuf::ReplaceStringFieldValueRecursive(value, realValue, "Config", &pbCfg, true);
        } else {
            throw yexception() << "Unknown substitution '" << value << "'" << Endl;
        }
    }
    return pbCfg;
}

}
