#include "tools.h"

#include <google/protobuf/text_format.h>
#include <google/protobuf/messagext.h>

#include "travel/hotels/proto/extensions/ext.pb.h"
#include "travel/proto/base_types/base_types.pb.h"

#include <mapreduce/yt/interface/protos/extension.pb.h>
#include <library/cpp/logger/global/global.h>
#include <library/cpp/protobuf/json/json2proto.h>
#include <util/stream/file.h>
#include <util/string/subst.h>
#include <util/generic/hash_set.h>
#include <contrib/libs/re2/re2/re2.h>

namespace {
    bool GetParseMessageAsJson(const google::protobuf::FieldDescriptor& field) {
        const auto& fieldOptions = field.options();
        return fieldOptions.HasExtension(NTravelProto::ParseMessageAsJson) && fieldOptions.GetExtension(NTravelProto::ParseMessageAsJson);
    }

    bool GetIsCgiKeyValue(const google::protobuf::FieldDescriptor& field) {
        const auto& fieldOptions = field.options();
        return fieldOptions.HasExtension(NTravelProto::CgiKeyValue) && fieldOptions.GetExtension(NTravelProto::CgiKeyValue);
    }

    const static re2::RE2 ConfigSubstitutionPattern = re2::RE2("^\\$\\{([A-Za-z0-9_:\\-]*)\\}$");
}

namespace NTravel {
namespace NProtobuf {

TString GetFieldName(const ::google::protobuf::FieldDescriptor* fd) {
    const ::google::protobuf::FieldOptions& options = fd->options();
    if (options.HasExtension(NYT::column_name)) {
        return options.GetExtension(NYT::column_name);
    }
    if (options.HasExtension(NYT::key_column_name)) {
        return options.GetExtension(NYT::key_column_name);
    }
    return fd->name();
}

NYT::TNode ProtoRecurseProcessRepeatedField(const ::google::protobuf::Message& proto, const ::google::protobuf::Reflection* r, const ::google::protobuf::FieldDescriptor* fd) {

    NYT::TNode list = NYT::TNode::CreateList();
    for (int idx = 0; idx < r->FieldSize(proto, fd); ++idx) {
        switch (fd->cpp_type()) {
            case ::google::protobuf::FieldDescriptor::CPPTYPE_INT32:
                list.AsList().push_back(r->GetRepeatedInt32(proto, fd, idx));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_INT64:
                list.AsList().push_back(r->GetRepeatedInt64(proto, fd, idx));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_UINT32:
                list.AsList().push_back(r->GetRepeatedUInt32(proto, fd, idx));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_UINT64:
                list.AsList().push_back(r->GetRepeatedUInt64(proto, fd, idx));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_FLOAT:
                list.AsList().push_back(r->GetRepeatedFloat(proto, fd, idx));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_DOUBLE:
                list.AsList().push_back(r->GetRepeatedDouble(proto, fd, idx));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_BOOL:
                list.AsList().push_back(r->GetRepeatedBool(proto, fd, idx));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_ENUM:
                list.AsList().push_back(r->GetRepeatedEnum(proto, fd, idx)->name());
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_STRING:
                list.AsList().push_back(r->GetRepeatedString(proto, fd, idx));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_MESSAGE:
                throw yexception() << "Repeated messages not supported yet";
        }
    }
    return list;
}

void ProtoRecurse(const ::google::protobuf::Message& proto, const TString& namePfx,  bool addEnumInt,
                  std::function<void(const TString& name, NYT::EValueType vt, bool hasValue, const NYT::TNode& data)> consumer) {
    auto d = proto.GetDescriptor();
    auto r = proto.GetReflection();

    for (int i = 0, end = d->field_count(); i < end; ++i) {
        const ::google::protobuf::FieldDescriptor* fd = d->field(i);
        TString fullName = namePfx;
        if (fullName) {
            fullName += "_";
        }
        fullName += GetFieldName(fd);

        if (fd->is_repeated()) {
            NYT::TNode node = ProtoRecurseProcessRepeatedField(proto, r, fd);
            consumer(fullName, NYT::VT_ANY, true, node);
            continue;
        }

        bool hasValue =  r->HasField(proto, fd);

        switch (fd->cpp_type()) {
            case ::google::protobuf::FieldDescriptor::CPPTYPE_INT32:
                consumer(fullName, NYT::VT_INT32, hasValue, r->GetInt32(proto, fd));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_INT64:
                consumer(fullName, NYT::VT_INT64, hasValue, r->GetInt64(proto, fd));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_UINT32:
                consumer(fullName, NYT::VT_UINT32, hasValue, r->GetUInt32(proto, fd));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_UINT64:
                consumer(fullName, NYT::VT_UINT64, hasValue, r->GetUInt64(proto, fd));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_FLOAT:
                consumer(fullName, NYT::VT_DOUBLE, hasValue, r->GetFloat(proto, fd));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_DOUBLE:
                consumer(fullName, NYT::VT_DOUBLE, hasValue, r->GetDouble(proto, fd));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_BOOL:
                consumer(fullName, NYT::VT_BOOLEAN, hasValue, r->GetBool(proto, fd));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_ENUM:
                consumer(fullName, NYT::VT_STRING, hasValue, r->GetEnum(proto, fd)->name());
                if (addEnumInt) {
                    consumer(fullName + "Int", NYT::VT_UINT64, hasValue, r->GetEnum(proto, fd)->number());
                }
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_STRING:
                consumer(fullName, NYT::VT_STRING, hasValue, r->GetString(proto, fd));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_MESSAGE:
                ProtoRecurse(r->GetMessage(proto, fd), fullName, addEnumInt, consumer);
                break;
        }
    }
}

NYT::TTableSchema GenerateTableSchema(const ::google::protobuf::Message& proto, bool addEnumInt) {
    NYT::TTableSchema schema;
    ProtoRecurse(proto, "", addEnumInt, [&schema](const TString& name, NYT::EValueType vt, bool hasValue, const NYT::TNode& data) {
        Y_UNUSED(hasValue, data);
        schema.AddColumn(name, vt);
    });
    return schema;
}

void ProtoToNode(const ::google::protobuf::Message& proto, NYT::TNode* node, bool addEnumInt) {
    ProtoRecurse(proto, "", addEnumInt, [&node](const TString& name, NYT::EValueType vt, bool hasValue, const NYT::TNode& data){
        Y_UNUSED(vt);
        if (hasValue) {
            (*node)[name] = data;
        } else {
            (*node)[name] = NYT::TNode::CreateEntity();
        }
    });
}

void NodeToProtoRecurseProcessRepeatedField(const TString& fullName, const NYT::TNode& node, const ::google::protobuf::Reflection* r, const ::google::protobuf::FieldDescriptor* fd, ::google::protobuf::Message* proto) {
    if (!node.IsList()) {
        throw yexception() << "Repeated field '" << fullName << "' expects list, got non-list";
    }
    for (const NYT::TNode& subNode: node.AsList()) {
        switch (fd->cpp_type()) {
            case ::google::protobuf::FieldDescriptor::CPPTYPE_INT32:
                r->AddInt32(proto, fd, subNode.IntCast<i32>());
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_INT64:
                r->AddInt64(proto, fd, subNode.IntCast<i64>());
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_UINT32:
                r->AddUInt32(proto, fd, subNode.IntCast<ui32>());
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_UINT64:
                r->AddUInt64(proto, fd, subNode.IntCast<ui64>());
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_DOUBLE:
                r->AddDouble(proto, fd, subNode.AsDouble());
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_FLOAT:
                r->AddFloat(proto, fd, subNode.AsDouble());
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_BOOL:
                r->AddBool(proto, fd, subNode.AsBool());
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_ENUM:
                if (subNode.IsString()) {
                    auto ev = fd->enum_type()->FindValueByName(subNode.AsString());
                    if (ev) {
                        r->AddEnum(proto, fd, ev);
                    } else {
                        throw yexception() << "Unknown string enum value " << subNode.AsString() << "' for field '" << fd->name() << "'";
                    }
                } else if (subNode.IsInt64() || subNode.IsUint64()) {
                    auto ev = fd->enum_type()->FindValueByNumber(subNode.IntCast<i64>());
                    if (ev) {
                        r->AddEnum(proto, fd, ev);
                    } else {
                        throw yexception() << "Unknown integer enum value " << subNode.IntCast<i64>() << "' for field '" << fd->name() << "'";
                    }
                }
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_STRING:
                r->AddString(proto, fd, subNode.AsString());
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_MESSAGE:
                throw yexception() << "Repeated submessages not supported in field '" << fullName << "'";
        }
    }
}

void NodeToProtoRecurse(const NYT::TNode& node, ::google::protobuf::Message* proto, const TString& namePfx) {
    auto d = proto->GetDescriptor();
    auto r = proto->GetReflection();

    for (int i = 0, end = d->field_count(); i < end; ++i) {
        const ::google::protobuf::FieldDescriptor* fd = d->field(i);
        TString fullName = namePfx;
        if (fullName) {
            fullName += "_";
        }
        fullName += GetFieldName(fd);
        if (!node.HasKey(fullName)) {
            continue;
        }
        const NYT::TNode& subNode = node[fullName];
        if (subNode.IsNull()) {
            continue;
        }
        if (fd->is_repeated()) {
            NodeToProtoRecurseProcessRepeatedField(fullName, subNode, r, fd, proto);
            continue;
        }

        switch (fd->cpp_type()) {
            case ::google::protobuf::FieldDescriptor::CPPTYPE_INT32:
                r->SetInt32(proto, fd, subNode.IntCast<i32>());
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_INT64:
                r->SetInt64(proto, fd, subNode.IntCast<i64>());
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_UINT32:
                r->SetUInt32(proto, fd, subNode.IntCast<ui32>());
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_UINT64:
                r->SetUInt64(proto, fd, subNode.IntCast<ui64>());
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_DOUBLE:
                r->SetDouble(proto, fd, subNode.AsDouble());
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_FLOAT:
                r->SetFloat(proto, fd, subNode.AsDouble());
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_BOOL:
                r->SetBool(proto, fd, subNode.AsBool());
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_ENUM:
                if (subNode.IsString()) {
                    auto ev = fd->enum_type()->FindValueByName(subNode.AsString());
                    if (ev) {
                        r->SetEnum(proto, fd, ev);
                    } else {
                        throw yexception() << "Unknown string enum value " << subNode.AsString() << "' for field '" << fd->name() << "'";
                    }
                } else if (subNode.IsInt64() || subNode.IsUint64()) {
                    auto ev = fd->enum_type()->FindValueByNumber(subNode.IntCast<i64>());
                    if (ev) {
                        r->SetEnum(proto, fd, ev);
                    } else {
                        throw yexception() << "Unknown integer enum value " << subNode.IntCast<i64>() << "' for field '" << fd->name() << "'";
                    }
                }
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_STRING:
                r->SetString(proto, fd, subNode.AsString());
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_MESSAGE:
                NodeToProtoRecurse(node, r->MutableMessage(proto, fd), fullName);
                break;
        }
    }
}

void NodeToProto(const NYT::TNode& node, ::google::protobuf::Message* proto) {
    NodeToProtoRecurse(node, proto, "");
}

void ParseTextFromFile(const TString& path, ::google::protobuf::Message* proto, bool merge) {
    const TString& data = TUnbufferedFileInput(path).ReadAll();

    ::google::protobuf::TextFormat::Parser parser;
    bool result;
    if (merge) {
        parser.AllowPartialMessage(true);
        result = parser.MergeFromString(data, proto);
    } else {
        result = parser.ParseFromString(data, proto);
    }
    if (!result) {
        ythrow yexception() << "can't parse protobuf from " << path;
    }
}

void AppendDelimitedToString(const ::google::protobuf::Message& message, TString& result) {
    TStringOutput output(result);
    ::google::protobuf::io::TCopyingOutputStreamAdaptor protobufAdaptor(&output);
    if (!::google::protobuf::io::SerializeToZeroCopyStreamSeq(&message, &protobufAdaptor)) {
          throw yexception() << "Error serializing message";
    }
}

void WalkStringFieldRecursive(const TString& fullPath, ::google::protobuf::Message* message, bool hideValueInLog, std::function<bool (const TString& fieldName, TString* fieldValue)> callback) {
    auto d = message->GetDescriptor();
    auto r = message->GetReflection();

    for (int i = 0, end = d->field_count(); i < end; ++i) {
        const ::google::protobuf::FieldDescriptor* fd = d->field(i);
        TString newFullPath = fullPath;
        if (newFullPath) {
            newFullPath += ".";
        }
        newFullPath += fd->name();
        switch (fd->cpp_type()) {
            case ::google::protobuf::FieldDescriptor::CPPTYPE_MESSAGE:
                // And we need to go deeper
                if (fd->is_repeated()) {
                    for (int i = 0; i < r->FieldSize(*message, fd); ++i) {
                        WalkStringFieldRecursive(newFullPath + "[" + ToString(i) + "]", r->MutableRepeatedMessage(message, fd, i), hideValueInLog, callback);
                    }
                } else {
                    if (r->HasField(*message, fd)) {
                        WalkStringFieldRecursive(newFullPath, r->MutableMessage(message, fd), hideValueInLog, callback);
                    }
                }
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_STRING:
                if (fd->is_repeated()) {
                    for (int i = 0; i < r->FieldSize(*message, fd); ++i) {
                        TString value = r->GetRepeatedString(*message, fd, i);
                        if (callback(fd->name(), &value)) {
                            INFO_LOG << "Change field " << newFullPath << "[" << ToString(i) << "] to " << (hideValueInLog ? "***" : value) << Endl;
                            r->SetRepeatedString(message, fd, i, value);
                        }
                    }
                } else {
                    TString value = r->GetString(*message, fd);
                    if (callback(fd->name(), &value)) {
                        INFO_LOG << "Change field " << newFullPath << " to " << (hideValueInLog ? "***" : value) << Endl;
                        r->SetString(message, fd, value);
                    }
                }
                break;
            default:
                break;
        }
    }
}

void SetExplicitDefaultValues(::google::protobuf::Message* proto) {
    auto d = proto->GetDescriptor();
    auto r = proto->GetReflection();

    for (int i = 0, end = d->field_count(); i < end; ++i) {
        const ::google::protobuf::FieldDescriptor* fd = d->field(i);
        if (fd->is_repeated()) {
            continue;
        }
        if (r->HasField(*proto, fd)) {
            continue;
        }

        switch (fd->cpp_type()) {
            case ::google::protobuf::FieldDescriptor::CPPTYPE_INT32:
                r->SetInt32(proto, fd, r->GetInt32(*proto, fd));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_INT64:
                r->SetInt64(proto, fd, r->GetInt64(*proto, fd));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_UINT32:
                r->SetUInt32(proto, fd, r->GetUInt32(*proto, fd));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_UINT64:
                r->SetUInt64(proto, fd, r->GetUInt64(*proto, fd));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_DOUBLE:
                r->SetDouble(proto, fd, r->GetDouble(*proto, fd));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_FLOAT:
                r->SetFloat(proto, fd, r->GetFloat(*proto, fd));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_BOOL:
                r->SetBool(proto, fd, r->GetBool(*proto, fd));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_ENUM:
                r->SetEnum(proto, fd, r->GetEnum(*proto, fd));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_STRING:
                r->SetString(proto, fd, r->GetString(*proto, fd));
                break;
            case ::google::protobuf::FieldDescriptor::CPPTYPE_MESSAGE:
                SetExplicitDefaultValues(r->MutableMessage(proto, fd));
                break;
        }
    }

}

void SetStringFieldRecursive(const TString& fieldName, const TString& value, const TString& fullPath, ::google::protobuf::Message* message, bool hideValueInLog) {
    WalkStringFieldRecursive(fullPath, message, hideValueInLog, [fieldName, value] (const TString& fieldNameActual, TString* fieldValue) {
        if (fieldNameActual != fieldName) {
            return false;
        }
        *fieldValue = value;
        return true;
    });
}

void ReplaceStringFieldValueRecursive(const TString& pattern, const TString& value, const TString& fullPath, ::google::protobuf::Message* message, bool hideValueInLog) {
    //NOTE that pattern and value are explicitly copied to avoid problems with SubstGlobal which checks for memory intersection
    TString patternCopy(pattern);
    TString valueCopy(value);
    patternCopy.Detach();
    valueCopy.Detach();
    WalkStringFieldRecursive(fullPath, message, hideValueInLog, [patternCopy, valueCopy] (const TString& fieldNameActual, TString* fieldValue) {
        Y_UNUSED(fieldNameActual);
        return SubstGlobal(*fieldValue, patternCopy, valueCopy) > 0;
    });
}

THashSet<std::pair<TString, TString>> FindAllConfigSubstitutions(const TString& fullPath, ::google::protobuf::Message* message) {
    THashSet<std::pair<TString, TString>> result;
    WalkStringFieldRecursive(fullPath, message, false, [&result](const TString& fieldNameActual, TString* fieldValue) {
        Y_UNUSED(fieldNameActual);
        TString matchResult;
        if (re2::RE2::FullMatch(*fieldValue, ConfigSubstitutionPattern, &matchResult)) {
            result.emplace(*fieldValue, matchResult);
        }
        return false;
    });
    return result;
}

template <typename TInt>
TInt IntFromString(TStringBuf value, const ::google::protobuf::FieldDescriptor* fd) {
    try {
        return FromString<TInt>(value);
    } catch (...) {
        const ::google::protobuf::FieldOptions& fieldOptions = fd->options();
        if (fieldOptions.HasExtension(NTravelProto::IntInvalidValue)) {
            return fieldOptions.GetExtension(NTravelProto::IntInvalidValue);
        }
        throw yexception() << "Invalid value '" << value << "' for int field '" << fd->name() << "': " << CurrentExceptionMessage();
    }
}

template <typename TOther>
TOther OtherFromString(TStringBuf value, const ::google::protobuf::FieldDescriptor* fd) {
    try {
        return FromString<TOther>(value);
    } catch (...) {
        throw yexception() << "Invalid value '" << value << "' for field '" << fd->name() << "': " << CurrentExceptionMessage();
    }
}

template <>
bool OtherFromString<bool>(TStringBuf value, const ::google::protobuf::FieldDescriptor* fd) {
    if (value == "1" || AsciiEqualsIgnoreCase(value, TStringBuf("true"))) {
        return true;
    } else if (value == "0" || AsciiEqualsIgnoreCase(value, TStringBuf("false"))) {
        return false;
    } else {
        throw yexception() << "Invalid value '" << value << "' for bool field '" << fd->name() << "'";
    }
}

template <>
const google::protobuf::EnumValueDescriptor* OtherFromString<const google::protobuf::EnumValueDescriptor*>(TStringBuf value, const ::google::protobuf::FieldDescriptor* f) {
    auto ev = f->enum_type()->FindValueByName(ToString(value));
    if (!ev) {
        int number;
        if (TryFromString(value, number)) {
            ev = f->enum_type()->FindValueByNumber(number);
        }
    }
    if (!ev) {
        throw yexception() << "Invalid enum value '" << value << "' for field '" << f->name() << "'";
    }
    return ev;
}

void CgiKeyValueToProto(const TStringBuf& formatted, NTravelProto::NBaseTypes::TStringPair* stringPair) {
    TString key;
    TString value;
    auto pos = formatted.find('=');
    if (pos == TStringBuf::npos) {
        key = formatted;
    } else {
        key = formatted.substr(0, pos);
        if (pos != formatted.length() - 1) {
            value = formatted.substr(pos + 1, formatted.length() - pos - 1);
        }
    }

    stringPair->SetKey(std::move(key));
    stringPair->SetValue(std::move(value));
}

void ProcessCgiField(const TCgiParameters& query, ::google::protobuf::Message* msg, int fieldNo, THashSet<TString>& seenFieldNames) {
    const ::google::protobuf::Descriptor* d = msg->GetDescriptor();
    auto r = msg->GetReflection();
    const auto f = d->field(fieldNo);

    if (f->cpp_type() == google::protobuf::FieldDescriptor::CPPTYPE_MESSAGE && !(GetParseMessageAsJson(*f) || GetIsCgiKeyValue(*f))) {
        Y_ENSURE(!f->is_repeated(), "Fields of type message can't be repeated (Bad field name: '" + f->name() + "')");
        auto subMsg = r->MutableMessage(msg, f);
        for (int i = 0, end = subMsg->GetDescriptor()->field_count(); i < end; ++i) {
            ProcessCgiField(query, subMsg, i, seenFieldNames);
        }
        return;
    }

    auto name = f->name();
    Y_ENSURE(seenFieldNames.insert(name).second, "Duplicate name '" + name + "'");
    if (f->options().HasExtension(NTravelProto::CgiName)) {
        auto cgiName = f->options().GetExtension(NTravelProto::CgiName);
        Y_ENSURE(seenFieldNames.insert(cgiName).second, "Duplicate name '" + cgiName + "'");
        if (query.contains(cgiName)) {
            Y_ENSURE(cgiName == name || !query.contains(name), "Both variants '" + cgiName + "' and '" + name + "' can't be used simultaneously");
            name = cgiName;
        }
    }

    if (GetIsCgiKeyValue(*f)) {
        Y_ENSURE(f->message_type() == NTravelProto::NBaseTypes::TStringPair::GetDescriptor(), "Wrong message type for CgiKeyValue. Expected TStringPair but got " + f->full_name());
    }

    auto pair = query.equal_range(name);
    bool haveValue = false;
    for (auto it = pair.first; it != pair.second; ++it) {
        TStringBuf v(it->second);
        if (v.empty() && (f->cpp_type() != google::protobuf::FieldDescriptor::CPPTYPE_STRING)) {
            // Пустые значения разрешим только для строк
            continue;
        }
        haveValue = true;
        switch (f->cpp_type()) {
#define PROCESS(CPP_TYPE, PROTO_ENUM, MEMBER, FS)                     \
            case google::protobuf::FieldDescriptor::PROTO_ENUM: {     \
                if (!v) {                                             \
                    throw yexception() << "Invalid empty for field '" \
                                       << name << "'";                \
                }                                                     \
                while (v) {                                           \
                    TStringBuf subVal = v.NextTok(',');               \
                    CPP_TYPE val = FS<CPP_TYPE>(subVal, f);           \
                    if (f->is_repeated()) {                           \
                        r->Add##MEMBER(msg, f, val);                  \
                    } else {                                          \
                        r->Set##MEMBER(msg, f, val);                  \
                    }                                                 \
                }                                                     \
                break;                                                \
            }
        PROCESS(bool, CPPTYPE_BOOL, Bool, OtherFromString);
        PROCESS(i32, CPPTYPE_INT32, Int32, IntFromString);
        PROCESS(i64, CPPTYPE_INT64, Int64, IntFromString);
        PROCESS(ui32, CPPTYPE_UINT32, UInt32, IntFromString);
        PROCESS(ui64, CPPTYPE_UINT64, UInt64, IntFromString);
        PROCESS(double, CPPTYPE_DOUBLE, Double, OtherFromString);
        PROCESS(float, CPPTYPE_FLOAT, Float, OtherFromString);
        PROCESS(const google::protobuf::EnumValueDescriptor*, CPPTYPE_ENUM, Enum, OtherFromString);
#undef PROCESS
            case google::protobuf::FieldDescriptor::CPPTYPE_STRING: {
                if (f->is_repeated()) {
                    r->AddString(msg, f, ToString(v));
                } else {
                    r->SetString(msg, f, ToString(v));
                }
                break;
            }
            case google::protobuf::FieldDescriptor::CPPTYPE_MESSAGE: {
                ::google::protobuf::Message* subMsg;
                if (f->is_repeated()) {
                    subMsg = r->AddMessage(msg, f);
                } else {
                    subMsg = r->MutableMessage(msg, f);
                }
                if (GetIsCgiKeyValue(*f)) {
                    auto stringPair = dynamic_cast<NTravelProto::NBaseTypes::TStringPair*>(subMsg);
                    Y_ENSURE(stringPair, "Wrong message type for CgiKeyValue. Expected TStringPair but got " + f->full_name());
                    CgiKeyValueToProto(v, stringPair);
                } else {
                    // сложный случай, будем парсить JSON
                    NProtobufJson::Json2Proto(v, *subMsg);
                }
                break;
            }
            default:
                break;
        }
    }
    if (!haveValue && !f->is_repeated()) {
        if (f->has_default_value()) {
            // Нет значения, но есть default, тогда надо его указать, чтобы логи были красивее (чтобы там не было null)
            switch (f->cpp_type()) {
#define PROCESS(CPP_TYPE, FIELD_NAME, NAME) case ::google::protobuf::FieldDescriptor::CPP_TYPE: r->Set##FIELD_NAME(msg, f, f->default_value##NAME()); break
                PROCESS(CPPTYPE_INT32, Int32, _int32);
                PROCESS(CPPTYPE_INT64, Int64, _int64);
                PROCESS(CPPTYPE_UINT32, UInt32, _uint32);
                PROCESS(CPPTYPE_UINT64, UInt64, _uint64);
                PROCESS(CPPTYPE_DOUBLE, Double, _double);
                PROCESS(CPPTYPE_FLOAT, Float, _float);
                PROCESS(CPPTYPE_BOOL, Bool, _bool);
                PROCESS(CPPTYPE_ENUM, Enum, _enum);
                PROCESS(CPPTYPE_STRING, String, _string);
#undef PROCESS
                default:
                    break;
            }
        } else {
            // Специальный случай для bool - проставляем false для тех случаев, когда нет default value
            if (f->cpp_type() ==  google::protobuf::FieldDescriptor::CPPTYPE_BOOL) {
                r->SetBool(msg, f, false);
            }
        }
    }
}

void ParseCgiRequest(const TCgiParameters& query, ::google::protobuf::Message* req) {
    auto seenFieldNames = THashSet<TString>();
    for (int i = 0, end = req->GetDescriptor()->field_count(); i < end; ++i) {
        ProcessCgiField(query, req, i, seenFieldNames);
    }
    if (!req->IsInitialized()) {
        throw yexception() << "Message of type \"" << req->GetDescriptor()->full_name()
                           << "\" is missing required fields: " << req->InitializationErrorString();
    }
}

TString NameGeneratorWithJsonName(const google::protobuf::FieldDescriptor& field) {
    const auto& fieldOptions = field.options();
    if (fieldOptions.HasExtension(NTravelProto::JsonName)) {
        return fieldOptions.GetExtension(NTravelProto::JsonName);
    } else {
        return field.name();
    }
}

TInstant TimestampToInstant(const google::protobuf::Timestamp& ts) {
    return TInstant::MicroSeconds(ts.seconds() * 1000000ull + ts.nanos() / 1000ull);
}

void InstantToTimestamp(TInstant instant, google::protobuf::Timestamp* ts) {
    ts->set_nanos((instant.MicroSeconds() % 1000000ull) * 1000);
    ts->set_seconds(instant.Seconds());
}
}// NProtobuf
}// NTravel
