#pragma once

#include <google/protobuf/message.h>
#include <google/protobuf/io/coded_stream.h>

#include <mapreduce/yt/interface/common.h>
#include <library/cpp/yson/node/node.h>
#include <library/cpp/cgiparam/cgiparam.h>

#include <google/protobuf/timestamp.pb.h>

#include <util/datetime/base.h>
#include <util/generic/string.h>

namespace NTravel {
namespace NProtobuf {

NYT::TTableSchema GenerateTableSchema(const ::google::protobuf::Message& proto, bool addEnumInt = false);

void ProtoToNode(const ::google::protobuf::Message& proto, NYT::TNode* node, bool addEnumInt = false);

void NodeToProto(const NYT::TNode& node, ::google::protobuf::Message* proto);

void ParseTextFromFile(const TString& path, ::google::protobuf::Message* proto, bool merge);

template <class T>
void ParseDelimitedFromString(const TString& data, TVector<T>* messages) {
    size_t sizeRest = data.size();
    const char * dataPtr = data.c_str();
    while (sizeRest) {
        ::google::protobuf::io::CodedInputStream input((const ui8*)dataPtr, sizeRest);
        // Read the size.
        ui32 partSize;
        if (!input.ReadVarint32(&partSize)) {
            throw yexception() << "Failed to read delimited size";
        }
        // Tell the stream not to read beyond that size.
        input.PushLimit(partSize);
        // Parse the message.
        messages->emplace_back();
        auto& message = messages->back();
        if (!message.MergeFromCodedStream(&input)) {
            throw yexception() << "Failed to parse message";
        };
        if (!input.ConsumedEntireMessage()) {
            throw yexception() << "Failed to finish message parsing";
        }
        sizeRest -= input.CurrentPosition();
        dataPtr += input.CurrentPosition();
    }
}

void AppendDelimitedToString(const ::google::protobuf::Message& message, TString& result);

template <class TId, class TProto>
TId GetIdentifier(const TProto& data);

template <class TId, class TProto>
TMap<TId, TProto> MergeOverrides(const ::google::protobuf::RepeatedPtrField<TProto>& records) {
    TMap<TId, TProto> result;
    for (const TProto& data: records) {
        auto& old = result[GetIdentifier<TId, TProto>(data)];
        old.MergeFrom(data);
    }
    return result;
}

template <class TId, class TProto>
TMap<TId, TProto> ReplaceOverrides(const ::google::protobuf::RepeatedPtrField<TProto>& records) {
    TMap<TId, TProto> result;
    for (const TProto& data: records) {
        auto& old = result[GetIdentifier<TId, TProto>(data)];
        old = data;
    }
    return result;
}

void SetExplicitDefaultValues(::google::protobuf::Message* message);

void SetStringFieldRecursive(const TString& fieldName, const TString& value, const TString& fullPath, ::google::protobuf::Message* message, bool hideValueInLog);

void ReplaceStringFieldValueRecursive(const TString& pattern, const TString& value, const TString& fullPath, ::google::protobuf::Message* message, bool hideValueInLog);

THashSet<std::pair<TString, TString>> FindAllConfigSubstitutions(const TString& fullPath, ::google::protobuf::Message* message);

void ParseCgiRequest(const TCgiParameters& query, ::google::protobuf::Message* req);

TString NameGeneratorWithJsonName(const google::protobuf::FieldDescriptor& field);

TInstant TimestampToInstant(const google::protobuf::Timestamp& ts);

void InstantToTimestamp(TInstant instant, google::protobuf::Timestamp* ts);

}// NProtobuf
}// NTravel
