#include "scheduler.h"

#include <travel/hotels/lib/cpp/util/profiletimer.h>

#include <library/cpp/logger/global/global.h>
#include <utility>

namespace NTravel {

void TScheduler::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    ct->insert(MAKE_COUNTER_PAIR(DutyNs));
    ct->insert(MAKE_COUNTER_PAIR(LatenessNs));
    ct->insert(MAKE_COUNTER_PAIR(NTasksInQueue));
}

bool TScheduler::TTask::operator < (const TTask& r) const {
    return NextTime > r.NextTime;
}

TScheduler::~TScheduler() {
    Stop();
}

void TScheduler::Start() {
    INFO_LOG << "Starting scheduler" << Endl;
    Thread_ = SystemThreadFactory()->Run(this);
    IsStarted_.Set();
}

void TScheduler::Stop() {
    if (!Thread_) {
        return;
    }
    StopFlag_.Set();
    WakeUp_.Signal();
    Thread_->Join();
    Thread_.Reset();
}

void TScheduler::RegisterCounters(NMonitor::TCounterSource& source) {
    source.RegisterSource(&Counters_, "Scheduler");
}

void TScheduler::Enqueue(TInstant at, std::function<void(void)> callback) {
    TTask task;
    task.NextTime = at;
    task.Callback = callback;
    task.IsPeriodical = false;
    EnqueueTask(task);
}

void TScheduler::Enqueue(TDuration after, TSchedulerCallback callback) {
    Enqueue(after.ToDeadLine(), callback);
}

void TScheduler::EnqueuePeriodical(TDuration period, TSchedulerCallback callback) {
    TTask task;
    task.NextTime = Now() + period;
    task.Callback = callback;
    task.IsPeriodical = true;
    task.Period = period;
    EnqueueTask(task);
}

void TScheduler::EnqueueTask(const TTask& task) {
    if (StopFlag_) {
        return;
    }
    Y_VERIFY(IsStarted_); // Scheduler MUST BE started before Enqueue
    with_lock (Lock_) {
        Counters_.NTasksInQueue.Inc();
        TInstant oldNextWakeUp = (Queue_.empty()) ? TInstant::Zero() : Queue_.top().NextTime;
        Queue_.push(task);
        if (!oldNextWakeUp || (task.NextTime < oldNextWakeUp)) {
            WakeUp_.Signal();
        }
    }
}

void TScheduler::DoExecute() {
    while (!StopFlag_) {
        TInstant nextTaskTime = GetNextTaskTime();
        if (!nextTaskTime) {
            WakeUp_.WaitI();
            continue;
        }
        if (nextTaskTime > Now()) {
            bool signaled = WakeUp_.WaitD(nextTaskTime);
            if (signaled) {
                continue;
            }
        }
        TProfileTimer dutyTimer;
        TTask task = PopNextTask();
        Counters_.LatenessNs += (Now() - task.NextTime).NanoSeconds();
        if (task.IsPeriodical) {
            EnqueuePeriodical(task.Period, task.Callback);
        }
        task.Callback();
        Counters_.DutyNs += dutyTimer.Get().NanoSeconds();
    }
}

TInstant TScheduler::GetNextTaskTime() const {
    TInstant nextTaskTime = TInstant::Zero();
    with_lock (Lock_) {
        if (!Queue_.empty()) {
            nextTaskTime = Queue_.top().NextTime;
        };
    }
    return nextTaskTime;
}

TScheduler::TTask TScheduler::PopNextTask() {
    TTask rec;
    with_lock (Lock_) {
        rec = Queue_.top();
        Queue_.pop();
    }
    Counters_.NTasksInQueue.Dec();
    return rec;
}

TScheduler& TScheduler::Instance() {
    return *Singleton<TScheduler>();
}

} // namespace NTravel
