#include "slack_forwarder_notificator.h"

#include <library/cpp/string_utils/url/url.h>
#include <library/cpp/logger/global/global.h>

namespace NTravel {

TString GetSchemeAndHost(const TString& url) {
    ui16 port;
    TStringBuf scheme, host;
    GetSchemeHostAndPort(url, scheme, host, port);
    return TString(scheme) + TString(host);
}

ui32 GetPort(const TString& url) {
    ui16 port;
    TStringBuf scheme, host;
    GetSchemeHostAndPort(url, scheme, host, port);
    return port;
}

TSlackForwarderNotificator::TSlackForwarderNotificator(const TString& url, const TString& componentName, ui64 taskId, ui64 revision)
    : ComponentName_(componentName)
    , TaskId_(taskId)
    , Revision_(revision)
    , HttpClient_(GetSchemeAndHost(url), GetPort(url))
{
}

TSlackForwarderNotificator::~TSlackForwarderNotificator() {

}

void TSlackForwarderNotificator::RegisterBuild(const TString& commitMessage, const TString& commiter) {
    NJsonWriter::TBuf json;
    json.BeginObject()
            .WriteKey("commit_message").WriteString(commitMessage)
            .WriteKey("committer").WriteString(commiter)
            .WriteKey("revision").WriteLongLong(Revision_)
        .EndObject();
    SendNotification("register-build", json);
}

void TSlackForwarderNotificator::RegisterBuildNoThrow(const TString& commitMessage, const TString& commiter) {
    try {
        RegisterBuild(commitMessage, commiter);
    } catch (...) {
        ERROR_LOG << "Failed to register build: " << CurrentExceptionMessage() << Endl;
    }
}

void TSlackForwarderNotificator::ReportStatus(const TString& status) {
    NJsonWriter::TBuf json;
    json.BeginObject()
            .WriteKey("revision").WriteLongLong(Revision_)
            .WriteKey("component_name").WriteString(ComponentName_)
            .WriteKey("status").WriteString(status)
            .WriteKey("task_id").WriteLongLong(TaskId_)
        .EndObject();
    SendNotification("report-status", json);
}

void TSlackForwarderNotificator::ReportStatusNoThrow(const TString& status) {
    try {
        ReportStatus(status);
    } catch (...) {
        ERROR_LOG << "Failed to report status: " << CurrentExceptionMessage() << Endl;
    }
}

void TSlackForwarderNotificator::SendNotification(const TString& type, const NJsonWriter::TBuf& data) {
    TString url = "/build-notifications/" + type;
    TString body = data.Str();
    TKeepAliveHttpClient::THeaders headers;
    headers["Content-Type"] = "application/json";
    auto code = HttpClient_.DoPost(url, body, nullptr, headers);
    if (code != 200) {
        throw yexception() << "Bad status code " << code << " at url " << url << ", body " << body;
    }
}

}//namespace NTravel
