#pragma once

#include <util/generic/vector.h>
#include <util/string/cast.h>
#include <util/generic/string.h>
#include <util/generic/variant.h>
#include <util/generic/yexception.h>
#include <util/system/types.h>

#include <type_traits>

namespace NTravel {
namespace NSQLite {

struct TExceptionError: yexception {
    using yexception::yexception;
};

struct TExceptionBusy: TExceptionError {
    using TExceptionError::TExceptionError;
};


struct TNone {};

struct TBlob: TString { // TString is suitable storage for binary data in Arcadia
    using TString::TString;
};

using TQueryParams = std::variant<TNone, i64, double, TString, TBlob>;

TString ToString(const TQueryParams& from);

namespace NDetail {

template <typename To>
struct TNumericVisitor {
    To operator()(const TNone&) const {
        ythrow TExceptionError{} << "NULL to number conversion is not supported";
    }

    To operator()(i64 from) const {
        return static_cast<To>(from);
    }

    To operator()(double from) const {
        return static_cast<To>(from);
    }

    To operator()(const TString& from) const {
        return FromString<To>(from);
    }

    To operator()(const TBlob&) const {
        ythrow TExceptionError{} << "BLOB to number conversion is not supported";
    }
};

} // NDetail

template <typename To>
To NumericCast(const TQueryParams& from) {
    static_assert(std::is_arithmetic<To>::value, "Only conversion to numeric types is supported");
    return std::visit(NDetail::TNumericVisitor<To>{}, from);
}

} // NSQLite
} // NTravel
