#include "db.h"

#include <library/cpp/logger/global/global.h>
#include <util/generic/variant.h>
#include <util/generic/vector.h>
#include <util/generic/string.h>
#include <util/generic/yexception.h>
#include <util/system/yassert.h>
#include <util/system/types.h>

namespace NTravel {
namespace NSQLite {

void TQuery::TStmtDeleter::Destroy(sqlite3_stmt* stmt) noexcept {
    if (sqlite3_finalize(stmt) != SQLITE_OK) {
        ERROR_LOG << "Failed to destroy sqlite3_stmt object" << Endl;
    }
}

TQuery::TQuery(TDatabase& db, const TString& sql, const TVector<TQueryParams>& params)
    : Db_(db)
{
    sqlite3_stmt* cstmt = nullptr;
    int result = sqlite3_prepare_v2(Db_.GetSQLiteDB(), sql.c_str(), int(sql.size()), &cstmt, nullptr);
    Stmt_.Reset(cstmt);
    Db_.Check(result);
    for (size_t i = 0; i < params.size(); ++i) {
        Bind(i, params[i]);
    }
    Step();
}

bool TQuery::Fetch(TVector<TQueryParams>* row) {
    if (IsDone()) {
        return false;
    }
    if (row) {
        int cols = sqlite3_column_count(Stmt_.Get());
        Y_ASSERT(0 <= cols);
        row->resize(size_t(cols));
        for (int i = 0; i < cols; ++i) {
            auto& r = (*row)[size_t(i)];
            switch (sqlite3_column_type(Stmt_.Get(), i)) {
                case SQLITE_INTEGER: {
                    r = i64(sqlite3_column_int64(Stmt_.Get(), i));
                    break;
                }
                case SQLITE_FLOAT: {
                    r = sqlite3_column_double(Stmt_.Get(), i);
                    break;
                }
                case SQLITE_TEXT: {
                    auto ptr = reinterpret_cast<const char*>(sqlite3_column_text(Stmt_.Get(), i));
                    auto size = size_t(sqlite3_column_bytes(Stmt_.Get(), i));
                    r = TString{ptr, size};
                    break;
                }
                case SQLITE_BLOB: {
                    auto ptr = reinterpret_cast<const char*>(sqlite3_column_blob(Stmt_.Get(), i));
                    auto size = size_t(sqlite3_column_bytes(Stmt_.Get(), i));
                    r = TBlob{ptr, size};
                    break;
                }
                default: {
                    r = TNone{};
                }
            }
        }
    }
    Step();
    return true;
}

size_t TQuery::Changes() const noexcept {
    return size_t(sqlite3_changes(Db_.GetSQLiteDB())); // it is safe to convert to unsigned
}

bool TQuery::IsDone() const noexcept {
    return !Stmt_;
}

struct TQuery::TBindVisitor {
    int operator()(const TNone&) {
        return sqlite3_bind_null(Stmt_, Order_);
    }

    int operator()(const i64& val) {
        return sqlite3_bind_int64(Stmt_, Order_, val);
    }

    int operator()(const double& val) {
        return sqlite3_bind_double(Stmt_, Order_, val);
    }

    int operator()(const TString& str) {
        return sqlite3_bind_text(Stmt_, Order_, str.data(), int(str.size()), SQLITE_TRANSIENT);
    }

    int operator()(const TBlob& blob) {
        return sqlite3_bind_blob(Stmt_, Order_, blob.data(), int(blob.size()), SQLITE_TRANSIENT);
    }

    sqlite3_stmt* Stmt_;
    int Order_;
};

void TQuery::Bind(size_t order, const TQueryParams& param) {
    Db_.Check(std::visit(TBindVisitor{Stmt_.Get(), int(order) + 1}, param));
}

void TQuery::Step() {
    int result = sqlite3_step(Stmt_.Get());
    switch (result) {
        case SQLITE_DONE:
            Stmt_.Reset();
            break;
        case SQLITE_ROW:
            break;
        default:
            Db_.Check(result);
    }
}

TTransaction::TTransaction(const TMutex& lock, TDatabase& db)
    : Guard_(Guard(lock))
    , Db_(db)
    , IsFinished_(false)
{
    Exec("BEGIN TRANSACTION");
}

TTransaction::~TTransaction() {
    if (IsFinished_) {
        return;
    }
    try {
        Rollback();
    } catch (const TExceptionError&) {
        ERROR_LOG << "Error during automatic transaction rollback: " << CurrentExceptionMessage() << Endl;
    }
}

THolder<TQuery> TTransaction::Exec(const TString& sql, const TVector<TQueryParams>& params) {
    if (IsFinished_) {
        throw TExceptionError() << "Cannot Exec() in already finished transaction";
    }
    return THolder(new TQuery{Db_, sql, params});
}

void TTransaction::Commit() {
    if (IsFinished_) {
        throw TExceptionError() << "Cannot Commit already finished transaction";
    }
    IsFinished_ = true;
    TQuery(Db_, "COMMIT");
}

void TTransaction::Rollback() {
    if (IsFinished_) {
        throw TExceptionError() << "Cannot Rollback already finished transaction";
    }
    IsFinished_ = true;
    TQuery(Db_, "ROLLBACK");
}

TDatabase::TDatabase(const TString& filePath, int flags) {
    sqlite3* db = nullptr;
    int result = sqlite3_open_v2(filePath.c_str(), &db, flags, nullptr);
    Db_.Reset(db);
    Check(result);
    Y_ASSERT(Db_);
}

TDatabase::~TDatabase() {
}

void TDatabase::ApplyPragma(const TString& pragma) {
    TQuery(*this, "PRAGMA " + pragma, {});
}

THolder<TTransaction> TDatabase::Transaction() {
    return THolder(new TTransaction(Lock_, *this));
}

void TDatabase::TDatabaseDeleter::Destroy(sqlite3* db) noexcept {
    if (sqlite3_close_v2(db) != SQLITE_OK) {
        ERROR_LOG << "Failed to destroy sqlite3 object, not all associated resources are deallocated" << Endl;
    }
}

TStringBuf TDatabase::ErrMsg() const noexcept {
    TStringBuf msg;
    if (Db_) {
        msg = sqlite3_errmsg(Db_.Get());
    }
    if (msg.empty()) {
        msg = "SQLite unknown error";
    }
    return msg;
}

void TDatabase::Check(int result) const {
    switch (result) {
        case SQLITE_OK:
            return;
        case SQLITE_BUSY:
        case SQLITE_LOCKED:
            ythrow TExceptionBusy{} << ErrMsg();
        default:
            ythrow TExceptionError{} << ErrMsg();
    }
}

sqlite3* TDatabase::GetSQLiteDB() const {
    return Db_.Get();
}


} // NSQLite
} // NTravel
