#pragma once

#include "data.h"

#include <contrib/libs/sqlite3/sqlite3.h>

#include <util/system/mutex.h>
#include <util/generic/noncopyable.h>
#include <util/generic/string.h>
#include <util/generic/strbuf.h>
#include <util/generic/vector.h>
#include <util/generic/ptr.h>

namespace NTravel {
namespace NSQLite {

class TDatabase;

class TQuery: TMoveOnly {
public:
    bool Fetch(TVector<TQueryParams>* row = nullptr);

    size_t Changes() const noexcept;
private:
    struct TStmtDeleter {
        static void Destroy(sqlite3_stmt* stmt) noexcept;
    };

    struct TBindVisitor;

    friend class TTransaction;
    friend class TDatabase;

    TDatabase& Db_;
    THolder<sqlite3_stmt, TStmtDeleter> Stmt_;

    TQuery(TDatabase& db, const TString& sql, const TVector<TQueryParams>& params = {});

    void Bind(size_t order, const TQueryParams& param);
    void Step();
    bool IsDone() const noexcept;
};

class TTransaction: TNonCopyable {
public:
    TTransaction(const TMutex& lock, TDatabase& db);
    ~TTransaction();

    THolder<TQuery> Exec(const TString& sql, const TVector<TQueryParams>& params = {});
    void Commit();
    void Rollback();
private:
    TGuard<TMutex> Guard_;
    TDatabase& Db_;
    bool IsFinished_;
};

class TDatabase: TNonCopyable {
public:
    TDatabase(const TString& filePath, int flags = (SQLITE_OPEN_READWRITE | SQLITE_OPEN_CREATE | SQLITE_OPEN_FULLMUTEX));
    ~TDatabase();

    void ApplyPragma(const TString& pragma);
    THolder<TTransaction> Transaction();
private:
    friend class TQuery;

    struct TDatabaseDeleter {
        static void Destroy(sqlite3* db) noexcept;
    };

    TMutex Lock_;
    THolder<sqlite3, TDatabaseDeleter> Db_;

    TStringBuf ErrMsg() const noexcept;
    void Check(int result) const;
    sqlite3* GetSQLiteDB() const;
};

} // NSQLite
} // NTravel
