#include "compress.h"

#include <contrib/libs/zstd/include/zstd.h>

#include <util/generic/buffer.h>
#include <util/stream/str.h>

namespace NTravel {

TString ZLibCompress(const TString& strIn, ZLib::StreamType type, size_t compressionLevel) {
    TString strOut;
    TStringInput input(strIn);
    TStringOutput output(strOut);
    TZLibCompress d(&output, type, compressionLevel);
    TransferData(&input, &d);
    d.Finish();
    output.Finish();
    return strOut;
}

TString ZLibDecompress(const TString& strIn, ZLib::StreamType type) {
    TString strOut;
    TStringInput input(strIn);
    TStringOutput output(strOut);
    TZLibDecompress d(&input, type);
    d.SetAllowMultipleStreams(false);
    TransferData(&d, &output);
    output.Finish();
    return strOut;
}

TString ZStdCompress(const TString& strIn, int compressionLevel) {
    size_t bufferSize = ZSTD_compressBound(strIn.size());
    TBuffer buffer(bufferSize);
    const auto resultSize = ZSTD_compress(
        buffer.Data(), bufferSize,
        strIn.data(), strIn.size(),
        compressionLevel
    );
    if (ZSTD_isError(resultSize) != 0) {
        throw yexception() << "ZSTD Compression failed: " << ZSTD_getErrorName(resultSize);
    }
    return TString(buffer.Data(), resultSize);
}

TString ZStdDecompress(const TString& strIn) {
    const auto size = ZSTD_getDecompressedSize(strIn.data(), strIn.size());
    TBuffer buffer(size);
    const auto decompressedSize = ZSTD_decompress(buffer.Data(), size, strIn.data(), strIn.size());
    if (ZSTD_isError(decompressedSize) != 0) {
        throw yexception() << "ZSTD Decompression failed: " << ZSTD_getErrorName(decompressedSize);
    }
    return TString(buffer.Data(), decompressedSize);
}

} // NTravel
