#pragma once

#include <type_traits>

#include <util/generic/hash_set.h>
#include <util/generic/vector.h>
#include <util/generic/yexception.h>
#include <util/system/mutex.h>
#include <util/system/guard.h>
#include <util/datetime/base.h>

#include <travel/hotels/lib/cpp/mon/counter.h>

namespace NTravel {

template<typename T>
typename std::enable_if<std::is_scalar<T>::value, i64>::type GetSize(const T&) {
    return sizeof(T);
}

template<typename T>
typename std::enable_if<std::is_same<T, TString>::value, i64>::type GetSize(const T& value) {
    return sizeof(T) + value.length();
}

template<typename T>
typename std::enable_if<std::is_same<T, TGUID>::value, i64>::type GetSize(const T&) {
    return sizeof(T);
}

struct THashSetWithTtlCounters : public NMonitor::TCounterSource {
    NMonitor::TCounter DeduplicationTotalBytes;
    NMonitor::TCounter DeduplicationTotalElements;

    void QueryCounters(NMonitor::TCounterTable* ct) const override {
        ct->insert(MAKE_COUNTER_PAIR(DeduplicationTotalBytes));
        ct->insert(MAKE_COUNTER_PAIR(DeduplicationTotalElements));
    }
};

template <typename T>
class THashSetWithTtl {
 public:
    explicit THashSetWithTtl(TDuration itemTtl, size_t generationsCount = 10)
        : GenerationTtl(generationsCount > 1 ? itemTtl / (generationsCount - 1) :
                        ythrow yexception() << "generationsCount should be greater than 1 (got " << generationsCount << ")")
        , Generations_(generationsCount)
        , CurrentGeneration_(0)
        , CurrentGenerationStart_(TInstant::Seconds(0)) {}

    bool Contains(const T& item, TInstant timestamp) {
        with_lock (Lock_) {
            MaybeRotateGenerations(timestamp);
            return Contains(item);
        }
    }

    bool Insert(const T& item, TInstant timestamp) {
        with_lock (Lock_) {
            MaybeRotateGenerations(timestamp);
            if (Contains(item)) {
                return false;
            }
            GetGeneration(0).AddItem(item);
            Counters_.DeduplicationTotalBytes += GetSize(item);
            Counters_.DeduplicationTotalElements++;
            return true;
        }
    }

    void RegisterCounters(NMonitor::TCounterSource& source, const TString& name) const {
        source.RegisterSource(&Counters_, name);
    }

    const THashSetWithTtlCounters& GetCounters() const {
        return Counters_;
    }

 private:

    class TGeneration {
    public:
        void AddItem(const T& item) {
            Items_.insert(item);
            TotalBytes_ += GetSize(item);
        }

        bool Contains(const T& item) const {
            return Items_.contains(item);
        }

        void Clear() {
            Items_.clear();
            TotalBytes_ = 0;
        }

        i64 Size() const {
            return Items_.size();
        }

        i64 GetTotalBytes() const {
            return TotalBytes_;
        }

    private:
        THashSet<T> Items_;
        i64 TotalBytes_;
    };

    const TDuration GenerationTtl;
    THashSetWithTtlCounters Counters_;

    TMutex Lock_;
    TVector<TGeneration> Generations_;
    size_t CurrentGeneration_;
    TInstant CurrentGenerationStart_;

    TGeneration& GetGeneration(size_t age) {
        return Generations_[(Generations_.size() + CurrentGeneration_ - age) % Generations_.size()];
    }

    bool Contains(const T& item) {
        for (size_t i = 0; i < Generations_.size(); i++) {
            if (GetGeneration(i).Contains(item)) {
                return true;
            }
        }
        return false;
    }

    void MaybeRotateGenerations(TInstant now) {
        if (CurrentGenerationStart_.Seconds() == 0) {
            CurrentGenerationStart_ = now;
            return;
        }
        while (now - CurrentGenerationStart_ > GenerationTtl) {
            CurrentGenerationStart_ += GenerationTtl;
            CurrentGeneration_++;

            auto& currGeneration = GetGeneration(0);
            Counters_.DeduplicationTotalBytes -= currGeneration.GetTotalBytes();
            Counters_.DeduplicationTotalElements -= currGeneration.Size();
            currGeneration.Clear();
        }
    }
};

}// NTravel
