#include <travel/hotels/lib/cpp/util/hash_set_with_ttl.h>

#include <library/cpp/testing/unittest/registar.h>
#include <util/thread/factory.h>

#include <iostream>

using namespace NTravel;

Y_UNIT_TEST_SUITE(TestHashSetWithTtl) {
    Y_UNIT_TEST(CheckSimple) {
        THashSetWithTtl<int> set(TDuration::Hours(3), 10);
        auto start = Now();
        UNIT_ASSERT_VALUES_EQUAL(set.Insert(1, start), true);
        UNIT_ASSERT_VALUES_EQUAL(set.Insert(1, start), false);
        UNIT_ASSERT_VALUES_EQUAL(set.Insert(1, start + TDuration::Seconds(1)), false);
    }

    Y_UNIT_TEST(CheckOldValuesAreDropped) {
        THashSetWithTtl<int> set(TDuration::Hours(3), 10);
        auto start = Now();
        for (int i = 0; i < 6 * 60; i++) {
            UNIT_ASSERT_VALUES_EQUAL(set.Insert(i, start + TDuration::Minutes(i)), true);
        }
        for (int i = 0; i < 2 * 60; i++) {
            UNIT_ASSERT_VALUES_EQUAL(set.Insert(i, start + TDuration::Minutes(i)), true);
        }
    }

    Y_UNIT_TEST(CheckDeduplicationStress) {
        THashSetWithTtl<int> set(TDuration::Hours(3), 10);

        auto threadCount = 5;
        auto totalIterations = 3 * 60 * 60;
        auto startSeconds = 0;

        auto start = Now();
        for (int i = 0; i < 100; i++) {
            TAtomicCounter counter;
            TVector<TAutoPtr<IThreadFactory::IThread>> threads;
            for (int th = 0; th < threadCount; th++) {
                threads.push_back(SystemThreadFactory()->Run([&set, &counter, totalIterations, start, startSeconds]() {
                    for (int seconds = startSeconds; seconds < startSeconds + totalIterations; seconds++) {
                        if (set.Insert(seconds, start + TDuration::Seconds(seconds))) {
                            counter.Inc();
                        }
                    }
                }));
            }
            for (int th = 0; th < 5; th++) {
                threads[th]->Join();
            }
            startSeconds += totalIterations;
            UNIT_ASSERT_VALUES_EQUAL_C(counter.Val(), totalIterations, std::to_string(i));
        }
    }

    Y_UNIT_TEST(CheckMetricsStress) {
        THashSetWithTtl<int> set(TDuration::Hours(3), 10);
        auto start = Now();
        int size = sizeof(int);
        for (int i = 0; i < 3 * 60; i++) {
            UNIT_ASSERT_VALUES_EQUAL(set.Insert(i, start + TDuration::Minutes(i)), true);
            UNIT_ASSERT_VALUES_EQUAL(set.GetCounters().DeduplicationTotalBytes.Val(), (i + 1) * size);
            UNIT_ASSERT_VALUES_EQUAL(set.GetCounters().DeduplicationTotalElements.Val(), i + 1);
        }
        for (int i = 3 * 60; i < 100 * 60; i++) {
            UNIT_ASSERT_VALUES_EQUAL(set.Insert(i, start + TDuration::Minutes(i)), true);
            for (int j = i - 3 * 60; j < i; j++) {
                UNIT_ASSERT_VALUES_EQUAL(set.Insert(i, start + TDuration::Minutes(i)), false);
            }
            UNIT_ASSERT_GE(set.GetCounters().DeduplicationTotalBytes.Val(), 3 * 60 * size);
            UNIT_ASSERT_GE(set.GetCounters().DeduplicationTotalElements.Val(), 3 * 60);
            UNIT_ASSERT_LE(set.GetCounters().DeduplicationTotalBytes.Val(), 4 * 60 * size);
            UNIT_ASSERT_LE(set.GetCounters().DeduplicationTotalElements.Val(), 4 * 60);
        }
    }
};
