#pragma once

#include <travel/hotels/lib/cpp/mon/counter_hypercube.h>
#include <travel/hotels/lib/cpp/util/sizes.h>

#include <util/generic/hash.h>
#include <util/generic/string.h>
#include <util/generic/vector.h>
#include <util/system/rwlock.h>

#include <utility>

namespace NTravel {
    class TObjectDeduplicator {
    private:
        template <class T>
        struct TStorageItem: public TThrRefBase {
            explicit TStorageItem(T value)
                : Value(std::move(value))
            {
            }

            size_t CalcTotalByteSize() const {
                return sizeof(TStorageItem<T>) + TTotalByteSize<T>()(Value) - sizeof(T);
            }

            T Value;
        };

        template <class T>
        struct TTypedStorage;

    public:
        template <class T>
        class TUnique {
        private:
            TIntrusivePtr<TStorageItem<T>> Value; // Using pointer instead of const reference to be able to construct empty TUnique<T> with any T and to count refs

            explicit TUnique(TIntrusivePtr<TStorageItem<T>> value)
                : Value(std::move(value))
            {
            }

            friend struct TTypedStorage<T>;

        public:
            TUnique()
                : Value(nullptr)
            {
            }

            bool IsEmpty() const {
                return Value == nullptr;
            }

            T GetValue() const {
                Y_ENSURE(!IsEmpty(), "Tried to GetValue of empty TUnique");
                return Value->Value;
            }

            bool operator==(const TUnique<T>& rhs) const {
                return Value.Get() == rhs.Value.Get(); // comparing pointers because different equal instances have same pointers
            }

            size_t Hash() const {
                return THash<TStorageItem<T>*>()(Value.Get());
            }

            size_t CalcTotalByteSize() const {
                return sizeof(TUnique<T>);
            }
        };

        explicit TObjectDeduplicator(const TString& name);
        void RegisterCounters(NMonitor::TCounterSource& source) const;

        template <class T>
        TUnique<T> Deduplicate(const TString& category, const T& value) {
            {
                TReadGuard g(Mutex_);
                auto it = Storages_.find(category);
                if (it != Storages_.end()) {
                    auto typedStorage = dynamic_cast<TTypedStorage<T>*>(it->second.get());
                    return typedStorage->Deduplicate(value);
                }
            }
            {
                TWriteGuard g(Mutex_);
                auto& storage = Storages_[category];
                if (!storage) {
                    storage = std::make_unique<TTypedStorage<T>>(Counters_.GetOrCreate({category}));
                }
                auto typedStorage = dynamic_cast<TTypedStorage<T>*>(storage.get());
                return typedStorage->Deduplicate(value);
            }
        }

        void RemoveUnusedRecords() {
            TVector<TStorage*> storages; // Saving pointers here to avoid long locks on Mutex_
            {
                TReadGuard g(Mutex_);
                storages.reserve(Storages_.size());
                for (auto& [_, storage] : Storages_) {
                    storages.push_back(storage.get());
                }
            }
            for (auto& storage : storages) {
                storage->RemoveUnusedRecords();
            }
        }

    private:
        struct TStorageCounters: public NMonitor::TCounterSource {
            NMonitor::TCounter NRecords;
            NMonitor::TCounter NBytes;

            void QueryCounters(NMonitor::TCounterTable* ct) const override;
        };

        struct TStorage {
            virtual void RemoveUnusedRecords() = 0;
            virtual ~TStorage() = default;
        };

        template <class T>
        struct TPointerHash {
            inline size_t operator()(const TIntrusivePtr<TStorageItem<T>>& v) const {
                return THash<T>()(v->Value);
            }
            inline size_t operator()(const T& v) const {
                return THash<T>()(v);
            }
        };

        template <class T>
        struct TPointerEqualTo {
            inline bool operator()(const TIntrusivePtr<TStorageItem<T>>& lhs, const TIntrusivePtr<TStorageItem<T>>& rhs) const {
                return TEqualTo<T>()(lhs->Value, rhs->Value);
            }
            inline bool operator()(const TIntrusivePtr<TStorageItem<T>>& lhs, const T& rhs) const {
                return TEqualTo<T>()(lhs->Value, rhs);
            }
        };

        template <class T>
        struct TTypedStorage: public TStorage {
            explicit TTypedStorage(NMonitor::TCounterHypercube<TStorageCounters>::TCountersRef counters)
                : Counters_(std::move(counters))
            {
            }

            TUnique<T> Deduplicate(const T& value) {
                {
                    TReadGuard g(Mutex_);
                    auto it = KnownObjects_.find(value);
                    if (it != KnownObjects_.end()) {
                        return TUnique<T>(*it);
                    }
                }
                TWriteGuard g(Mutex_);
                auto [it, inserted] = KnownObjects_.emplace(MakeIntrusive<TStorageItem<T>>(value));
                if (inserted) {
                    UpdateRecordCounts(*it, 1);
                }
                return TUnique<T>(*it);
            }

            void RemoveUnusedRecords() override {
                TWriteGuard g(Mutex_); // Actions under this Mutex_ are background, so it's ok to block them for a while
                for (auto it = KnownObjects_.begin(); it != KnownObjects_.end();) {
                    if (it->RefCount() == 1) {
                        UpdateRecordCounts(*it, -1);
                        KnownObjects_.erase(it++);
                    } else {
                        ++it;
                    }
                }
            }

            void UpdateRecordCounts(const TIntrusivePtr<TStorageItem<T>> item, int sign) {
                Counters_->NRecords += sign;
                Counters_->NBytes += (sizeof(typename decltype(KnownObjects_)::value_type) + TTotalByteSize<TStorageItem<T>>()(*item)) * sign;
            }

            TRWMutex Mutex_;
            THashSet<TIntrusivePtr<TStorageItem<T>>, TPointerHash<T>, TPointerEqualTo<T>> KnownObjects_;
            NMonitor::TCounterHypercube<TStorageCounters>::TCountersRef Counters_;
        };

        const TString Name_;
        TRWMutex Mutex_;
        THashMap<TString, std::unique_ptr<TStorage>> Storages_;
        NMonitor::TCounterHypercube<TStorageCounters> Counters_;
    };
}

template <class T>
struct THash<NTravel::TObjectDeduplicator::TUnique<T>> {
    inline size_t operator()(const NTravel::TObjectDeduplicator::TUnique<T>& v) const {
        return v.Hash();
    }
};
