#include "restart_detector.h"

#include <library/cpp/logger/global/global.h>
#include <google/protobuf/text_format.h>

#include <util/stream/file.h>

namespace NTravel {

TRestartDetector::TRestartDetector(const TString& stateFilePath)
    : Path(stateFilePath)
{
    LoadState();
    if (!State_.GetShutdownCalled()) {
        State_.SetSuspiciousStartCount(State_.GetSuspiciousStartCount() + 1);
    } else {
        State_.SetSuspiciousStartCount(0);
    }
    State_.SetShutdownCalled(false);
    SaveState();

    SuspiciousStartsCount = State_.GetSuspiciousStartCount();
    StartTime_ = Now();
}

void TRestartDetector::ReportShutdown() {
    State_.SetShutdownCalled(true);
    SaveState();
}

void TRestartDetector::QueryCounters(NMonitor::TCounterTable* ct) const {
    if (SuspiciousStartsCount.Val() > 0 && Now() - StartTime_ > TDuration::Minutes(3)) {
        SuspiciousStartsCount = 0;
    }
    ct->insert(MAKE_COUNTER_PAIR(SuspiciousStartsCount));
}

void TRestartDetector::RegisterCounters(NMonitor::TCounterSource& counters) const {
    counters.RegisterSource(this, "RestartDetector");
}

void TRestartDetector::LoadState() {
    try {
        if (!Path.Exists()) {
            State_.SetShutdownCalled(true);
            State_.SetSuspiciousStartCount(0);
            return;
        }
        auto data = TUnbufferedFileInput(Path).ReadAll();
        ::google::protobuf::TextFormat::Parser parser;
        if (parser.ParseFromString(data, &State_)) {
            return;
        }
    } catch (...) {
        WARNING_LOG << "Failed to load restart detector state from " << Path << ": " << CurrentExceptionMessage() << Endl;
    }
    State_.SetShutdownCalled(false);
    State_.SetSuspiciousStartCount(0);
}

void TRestartDetector::SaveState() const {
    try {
        TString result;
        Path.Parent().MkDirs();
        ::google::protobuf::TextFormat::PrintToString(State_, &result);
        TUnbufferedFileOutput(Path).Write(result);
    } catch (...) {
        WARNING_LOG << "Failed to save restart detector state to " << Path << ": " << CurrentExceptionMessage() << Endl;
    }
}

}// NTravel
