#include "signals.h"

#include <util/stream/output.h>
#include <util/string/printf.h>
#include <util/system/backtrace.h>

#ifdef _unix_
#   include <signal.h>
#   include <stdlib.h>
#endif

#ifdef _unix_
static struct sigaction def_sigsegv_action;

static void SignalHandler(int signal, siginfo_t* info, void*) {
    Cerr << Sprintf("signal caught: %s", strsignal(signal)) << Endl;

    if (signal ==  SIGSEGV) {
        const char* reason;

        switch (info->si_code) {
            case SEGV_MAPERR:
                reason = "address not mapped to object";
                break;
            case SEGV_ACCERR:
                reason = "invalid permissions for mapped object";
                break;
            default:
                reason = "unknown";
                break;
        }

        Cerr << Sprintf("faulting address: %p, code %d (%s)", info->si_addr, info->si_code, reason) << Endl;
        PrintBackTrace();
        //! Call default handler to produce coredump
        (*def_sigsegv_action.sa_handler)(signal);
    }

    exit(1);
}

void SetupSignalHandlers() {
    struct sigaction sa = {};

    sa.sa_handler   = SIG_IGN;
    sigaction(SIGPIPE, &sa, nullptr);

    sa.sa_sigaction = SignalHandler;
    sa.sa_flags     = SA_SIGINFO;
    sigaction(SIGSEGV, &sa, &def_sigsegv_action);
}

#else
void SetupSignalHandlers() { }
#endif // _unix_
