#include "string_compressor.h"

#include "sizes.h"

#include <util/string/builder.h>

namespace {
    struct TAutoRunningStateReporter {
        explicit TAutoRunningStateReporter(NMonitor::TCounter* counter)
            : Counter(counter)
        {
            Counter->Inc();
        }

        ~TAutoRunningStateReporter() {
            Counter->Dec();
        }

        NMonitor::TCounter* Counter;
    };

    struct TAutoSpentMillisecondsReported {
        explicit TAutoSpentMillisecondsReported(NMonitor::TCounter* counter)
            : Counter(counter)
        {
        }

        ~TAutoSpentMillisecondsReported() {
            *Counter = Timer.Get().MilliSeconds();
        }

        TProfileTimer Timer;
        NMonitor::TCounter* Counter;
    };
}

namespace NTravel {
    TStringCompressorRebuildStrategy::TStringCompressorRebuildStrategy(
        double sampleProb,
        size_t samplesForInitialTrain,
        TMaybe<TDuration> rebuildEachPeriod,
        TMaybe<size_t> rebuildEachNSamples,
        TMaybe<size_t> maxTrainStringsPerRebuild)
        : SampleProb_(sampleProb)
        , SamplesForInitialTrain_(samplesForInitialTrain)
        , TrainSamplesCount_(0)
        , FirstTrainTriggered_(false)
        , RebuildTrigger_(nullptr)
        , RebuidEachPeriod_(rebuildEachPeriod)
        , RebuildEachNSamples_(rebuildEachNSamples)
        , MaxTrainStringsPerRebuild_(maxTrainStringsPerRebuild)
    {
    }

    void TStringCompressorRebuildStrategy::Start() {
        TGuard guard(Mutex_);
        if (RebuidEachPeriod_.Defined()) {
            RebuildTriggeringThread_ = SystemThreadFactory()->Run([this]() {
                while (!Stopping_) {
                    with_lock (Mutex_) {
                        if (FirstTrainTriggered_) {
                            TriggerRebuild();
                        }
                    }
                    StopEvent_.WaitT(*RebuidEachPeriod_.Get());
                }
            });
        }
    }

    void TStringCompressorRebuildStrategy::Stop() {
        if (!Stopping_.TrySet()) {
            return;
        }
        StopEvent_.Signal();
        THolder<IThreadFactory::IThread>* thread = nullptr;
        with_lock (Mutex_) {
            if (!RebuildTriggeringThread_) {
                return;
            }
            thread = &RebuildTriggeringThread_;
        }
        (*thread)->Join();
        with_lock (Mutex_) {
            RebuildTriggeringThread_ = nullptr;
        }
    }

    void TStringCompressorRebuildStrategy::SetRebuildTrigger(std::function<void()> function) {
        TGuard guard(Mutex_);
        RebuildTrigger_ = std::move(function);
    }

    bool TStringCompressorRebuildStrategy::ShouldUseStringForTrain(const TString& value) {
        auto hash = THash<TString>()(value);
        auto shouldUseByHash = (hash % Max(1, static_cast<int>(1 / (SampleProb_ + SampleProbEps_)))) == 0;
        with_lock (Mutex_) {
            if (MaxTrainStringsPerRebuild_.Defined() && TrainSamplesCount_ >= *MaxTrainStringsPerRebuild_.Get()) {
                return false;
            }
            auto shouldUse = !FirstTrainTriggered_ || shouldUseByHash;
            TMaybe<size_t> needSamples = FirstTrainTriggered_ ? RebuildEachNSamples_ : SamplesForInitialTrain_;
            if (shouldUse) {
                TrainSamplesCount_++;
            }
            if (TrainSamplesCount_ == needSamples) {
                TriggerRebuild();
            }
            return shouldUse;
        }
    }

    void TStringCompressorRebuildStrategy::RebuildDone(bool) {
    }

    TStringCompressorRebuildStrategy::~TStringCompressorRebuildStrategy() {
        Stop();
    }

    void TStringCompressorRebuildStrategy::TriggerRebuild() {
        if (RebuildTrigger_) {
            RebuildTrigger_();
        }
        FirstTrainTriggered_ = true;
        TrainSamplesCount_ = 0;
    }

    const char* TCompressedString::DangerouslyGetInnerDataOnlyForTests() const {
        return std::get<TInnerDataPtr>(Data)->Info.load().Start;
    }

    TCompressedString::TCompressedString(const TInnerDataPtr& value)
        : Data(value)
    {
    }

    TCompressedString::TCompressedString()
        : TCompressedString(TInnerDataPtr())
    {
    }

    TCompressedString::TCompressedString(const TString& value)
        : Data(value)
    {
    }

    size_t TCompressedString::CalcTotalByteSize() const {
        if (std::holds_alternative<TInnerDataPtr>(Data)) {
            return sizeof(TCompressedString);
        } else {
            return sizeof(TCompressedString) + TTotalByteSize<TString>()(std::get<TString>(Data)) - sizeof(TString);
        }
    }

    size_t TCompressedString::Hash() const {
        if (std::holds_alternative<TInnerDataPtr>(Data)) {
            return THash<TInnerDataPtr>()(std::get<TInnerDataPtr>(Data));
        } else {
            return THash<TString>()(std::get<TString>(Data));
        }
    }

    bool TCompressedString::operator==(const TCompressedString& other) const {
        if (std::holds_alternative<TInnerDataPtr>(Data)) {
            Y_ENSURE(std::holds_alternative<TInnerDataPtr>(other.Data), "Tried to compare two strings from different TStringCompressors");
            return std::get<TInnerDataPtr>(Data).Get() == std::get<TInnerDataPtr>(other.Data).Get(); // Comparing pointers because strings are deduplicated
        } else {
            Y_ENSURE(std::holds_alternative<TString>(other.Data), "Tried to compare two strings from different TStringCompressors");
            return std::get<TString>(Data) == std::get<TString>(Data);
        }
    }

    bool TCompressedString::operator!=(const TCompressedString& other) const {
        return !(*this == other);
    }

    bool TCompressedString::IsDefined() const {
        return !std::holds_alternative<TInnerDataPtr>(Data) || std::get<TInnerDataPtr>(Data) != nullptr;
    }

    TStringCompressor::TStringCompressor(TString name,
                                         bool enabled,
                                         ui32 compressionLevel,
                                         double trainSamplingProbability,
                                         size_t samplesForInitialTrainCount,
                                         TMaybe<TDuration> rebuildEachPeriod,
                                         TMaybe<size_t> rebuildEachNSamples,
                                         TMaybe<size_t> maxTrainStringsPerRebuild,
                                         TMaybe<TDuration> previousGenerationCleanupDelay)
        : Name_(std::move(name))
        , Enabled_(enabled)
        , CompressionLevel_(compressionLevel)
        , PreviousGenerationCleanupDelay_(previousGenerationCleanupDelay.GetOrElse(TDuration::Minutes(1)))
        , Strategy_(trainSamplingProbability, samplesForInitialTrainCount, rebuildEachPeriod, rebuildEachNSamples, maxTrainStringsPerRebuild)
        , CurrentGenerationIndex_(0)
        , CurrentGeneration_(MakeAtomicShared<TGeneration>(Name_, CompressionLevel_, CurrentGenerationIndex_, TVector<TString>()))
        , CurrentCompressedStrings_(CompressedStringsBucketCount)
        , CurrentCompressedStringsByteSize_(static_cast<size_t>(CurrentCompressedStrings_.capacity() * sizeof(decltype(CurrentCompressedStrings_)::value_type)))
        , CurrentGenerationRawPtr_(CurrentGeneration_.Get())
        , PreviousGenerationRawPtr_(nullptr)
        , Counters_()
    {
        Strategy_.SetRebuildTrigger([this]() {
            StartRebuild();
        });
    }

    TStringCompressor::TStringCompressor(TString name, const NTravelProto::NAppConfig::TConfigStringCompressor& config)
        : TStringCompressor(std::move(name),
                            config.GetEnabled(),
                            config.GetCompressionLevel(),
                            config.GetTrainSamplingProbability(),
                            config.GetSamplesForInitialTrainCount(),
                            config.HasRebuildEachPeriodSec() ? TDuration::Seconds(config.GetRebuildEachPeriodSec()) : TMaybe<TDuration>(),
                            config.HasRebuildEachNSamples() ? config.GetRebuildEachNSamples() : TMaybe<size_t>(),
                            config.HasMaxTrainStringsPerRebuild() ? config.GetMaxTrainStringsPerRebuild() : TMaybe<size_t>(),
                            TMaybe<TDuration>()) {
    }

    void TStringCompressor::Start() {
        if (!Enabled_) {
            return;
        }
        if (Started_.TrySet()) {
            Strategy_.Start();
            RebuildThread_ = SystemThreadFactory()->Run([this]() { RunRebuildThread(); });
        } else {
            WARNING_LOG << "Duplicate call of Start() (" << Name_ << ")" << Endl;
        }
    }

    void TStringCompressor::Stop() {
        if (!Enabled_) {
            return;
        }
        if (!Started_) {
            WARNING_LOG << "Call of Stop() before Start() (" << Name_ << ")" << Endl;
            return;
        }
        if (!Stopping_.TrySet()) {
            return;
        }
        Strategy_.Stop();
        if (RebuildThread_) {
            RebuildEvent_.Signal();
            CleanupEvent_.Signal();
            RebuildThread_->Join();
            RebuildThread_ = nullptr;
        }
    }

    void TStringCompressor::RegisterCounters(NMonitor::TCounterSource& source) {
        if (!Enabled_) {
            return;
        }
        source.RegisterSource(&Counters_, Name_);
    }

    TCompressedString TStringCompressor::Compress(const TString& value) {
        if (!Enabled_) {
            return TCompressedString(value);
        }
        auto valueHash = THash<TString>()(value);
        auto bucket = valueHash % CurrentCompressedStrings_.size();
        with_lock (Mutex_) {
            UpdateMetricsBeforeDeduplication(value, 1);
            for (const auto& [currHash, currItem] : CurrentCompressedStrings_[bucket]) {
                if (currHash != valueHash) {
                    continue;
                }
                auto decompressed = DoDecompressUnlocked(TCompressedString(currItem));
                Y_ENSURE(decompressed.Defined(), "Failed to decompress string even under lock");
                if (*decompressed.Get() == value) {
                    return TCompressedString(currItem);
                }
            }
        }
        MaybeUseStringForTrain(value);
        with_lock (Mutex_) {
            // Lock is required here to atomically compress value (to avoid compressions using old CurrentGeneration_)
            auto ptr = MakeIntrusive<TCompressedString::TInnerData>(CurrentGeneration_->Compress(value));
            CurrentCompressedStrings_[bucket].emplace_back(valueHash, ptr);
            UpdateCurrentCompressedStringsByteSize(1);
            UpdateMetricsAfterCompress(value);
            return TCompressedString(ptr);
        }
    }

    TString TStringCompressor::Decompress(const TCompressedString& value) const {
        if (!Enabled_) {
            return std::get<TString>(value.Data);
        }
        auto result = DoDecompressUnlocked(value);
        if (result.Defined()) {
            return *result.Get();
        }
        with_lock (Mutex_) {
            result = DoDecompressUnlocked(value);
            Y_ENSURE(result.Defined(), "Failed to decompress string even under lock");
            return *result.Get();
        }
    }

    size_t TStringCompressor::GetStringAllocSizeOnlyForTests(const TCompressedString& value) const {
        if (!Enabled_) {
            return TTotalByteSize<TString>()(std::get<TString>(value.Data));
        }
        auto info = std::get<TCompressedString::TInnerDataPtr>(value.Data)->Info.load();
        return info.GetLength() + sizeof(TCompressedString::TInnerDataPtr::TValueType);
    }

    void TStringCompressor::OnTrainFinish(std::function<void(bool)> callback) {
        TrainFinishCallback_ = std::move(callback);
    }

    size_t TStringCompressor::GetStringCount() const {
        with_lock (Mutex_) {
            return CurrentGeneration_->StringCount;
        }
    }

    void TStringCompressor::TriggerRebuild() {
        StartRebuild();
    }

    TStringCompressor::~TStringCompressor() {
        Stop();
    }

    void TStringCompressor::StartRebuild() {
        RebuildEvent_.Signal();
    }

    void TStringCompressor::RunRebuildThread() {
        while (!Stopping_) {
            RebuildEvent_.Wait();
            if (Stopping_) {
                break;
            }
            {
                TAutoRunningStateReporter isRunningCounter(&Counters_.RebuildIsRunning);
                TAutoSpentMillisecondsReported timeCounter(&Counters_.LastRebuildTimeSpentMs);
                TVector<TString> trainingStrings;

                with_lock (TrainingStringsMutex_) {
                    std::swap(trainingStrings, TrainingStrings_);
                    TrainingStrings_.reserve(trainingStrings.size());
                }
                with_lock (Mutex_) {
                    CurrentGenerationIndex_++;
                    CurrentGenerationIndex_ %= 32; // 5 bytes of TCompressedString::TCompressedStringMeta::Gen
                }

                auto localGeneration = MakeAtomicShared<TGeneration>(Name_, CompressionLevel_, CurrentGenerationIndex_, trainingStrings);
                auto succeed = localGeneration->IsSuccessfulTrain();

                with_lock (Mutex_) {
                    std::swap(CurrentGeneration_, localGeneration);
                    PreviousGenerationRawPtr_.store(localGeneration.Get());
                    CurrentGenerationRawPtr_.store(CurrentGeneration_.Get());

                    Counters_.NPreviousGenerationBytes = Counters_.NCurrGenerationBytes;
                    Counters_.NPreviousGenerationChars = Counters_.NCurrGenerationChars;

                    Counters_.NCurrGenerationStringsBeforeDeduplication = 0;
                    Counters_.NCurrGenerationIncomeBytesBeforeDeduplication = 0;
                    Counters_.NCurrGenerationIncomeCharsBeforeDeduplication = 0;
                    Counters_.CompressionRatioBytesBeforeDeduplication = 0;
                    Counters_.CompressionRatioCharsBeforeDeduplication = 0;

                    Counters_.NCurrGenerationStrings = 0;
                    Counters_.NCurrGenerationBytes = 0;
                    Counters_.NCurrGenerationChars = 0;
                    Counters_.NCurrGenerationIncomeBytes = 0;
                    Counters_.NCurrGenerationIncomeChars = 0;

                    Counters_.NNextTrainStringsBytes = 0;
                    Counters_.NStringsForNextTrain = 0;
                    Counters_.NStringsForLastTrain = trainingStrings.size();

                    Counters_.CompressionRatioBytes = 0;
                    Counters_.CompressionRatioChars = 0;
                }

                for (auto& compressedStrings : CurrentCompressedStrings_) {
                    if (Stopping_) {
                        return;
                    }
                    while (true) {
                        std::pair<size_t, TCompressedString::TInnerDataPtr> compressedString;
                        with_lock (Mutex_) {
                            if (compressedStrings.empty()) {
                                break;
                            }
                            compressedString = compressedStrings.front();
                            if (compressedString.second.Get()->Info.load().GetGeneration() == CurrentGenerationIndex_) {
                                break;
                            }
                            compressedStrings.pop_front();
                        }
                        auto refcount = compressedString.second.RefCount();
                        if (refcount == 1) {
                            UpdateCurrentCompressedStringsByteSize(-1);
                            continue;
                        }
                        auto info = compressedString.second->Info.load();
                        if (localGeneration.Get()->Generation != info.GetGeneration()) {
                            ERROR_LOG << "Generation mismatch: " << localGeneration.Get()->Generation << " != " << info.GetGeneration() << Endl;
                            with_lock (Mutex_) {
                                ERROR_LOG << "Compressed strings size: " << compressedStrings.size() << Endl;
                                TStringBuilder s;
                                for (const auto& x: compressedStrings) {
                                    s << " " << x.second->Info.load().GetLength() << ":" << x.second->Info.load().GetGeneration() << " ";
                                }
                                ERROR_LOG << "Compressed strings data: " << s << Endl;
                            }
                        }
                        Y_ENSURE(localGeneration.Get()->Generation == info.GetGeneration());
                        auto decompressed = localGeneration.Get()->Decompress(info);
                        MaybeUseStringForTrain(decompressed);
                        with_lock (Mutex_) {
                            UpdateMetricsBeforeDeduplication(decompressed, refcount - 1);
                            compressedString.second->Info.store(CurrentGeneration_->Compress(decompressed));
                            UpdateMetricsAfterCompress(decompressed);
                            compressedStrings.emplace_back(std::move(compressedString));
                        }
                    }
                }

                with_lock (Mutex_) {
                    PreviousGenerationRawPtr_.store(nullptr);
                }

                Strategy_.RebuildDone(succeed);
                if (TrainFinishCallback_) {
                    TrainFinishCallback_(succeed);
                }

                if (Stopping_) {
                    break;
                }
                CleanupEvent_.WaitT(PreviousGenerationCleanupDelay_); // waiting for all clients before destructing localGeneration (which holds previous generation)
                with_lock (Mutex_) {
                    Counters_.NPreviousGenerationBytes = 0;
                    Counters_.NPreviousGenerationChars = 0;
                }
            }
        }
    }
    void TStringCompressor::MaybeUseStringForTrain(const TString& value) {
        if (!Strategy_.ShouldUseStringForTrain(value)) {
            return;
        }
        Counters_.NStringsForNextTrain.Inc();
        Counters_.NNextTrainStringsBytes += TTotalByteSize<TString>()(value);
        with_lock (TrainingStringsMutex_) {
            TrainingStrings_.emplace_back(value);
        }
    }

    TMaybe<TString> TStringCompressor::DoDecompressUnlocked(const TCompressedString& value) const {
        const auto& innerData = std::get<TCompressedString::TInnerDataPtr>(value.Data);
        Y_ENSURE(innerData, "Tried to decompress empty TCompressedString");
        auto info = innerData->Info.load();
        for (auto atomicRawPtr : {&CurrentGenerationRawPtr_, &PreviousGenerationRawPtr_}) {
            auto generationRawPtr = atomicRawPtr->load();
            if (generationRawPtr && generationRawPtr->Generation == info.GetGeneration()) {
                return generationRawPtr->Decompress(info);
            }
        }
        return {};
    }

    void TStringCompressor::UpdateMetricsAfterCompress(const TString& value) {
        Counters_.NCurrGenerationStrings = CurrentGeneration_->StringCount;
        Counters_.NCurrGenerationIncomeBytes += TTotalByteSize<TString>()(value);
        Counters_.NCurrGenerationIncomeChars += value.length();
        Counters_.NCurrGenerationBytes = CurrentGeneration_->TotalBytes + CurrentCompressedStringsByteSize_.Val();
        Counters_.NCurrGenerationChars = CurrentGeneration_->TotalStringLength;
        Counters_.CompressionRatioBytes = static_cast<double>(Counters_.NCurrGenerationIncomeBytes) / Counters_.NCurrGenerationBytes;
        Counters_.CompressionRatioChars = static_cast<double>(Counters_.NCurrGenerationIncomeChars) / Counters_.NCurrGenerationChars;

        Counters_.CompressionRatioBytesBeforeDeduplication = static_cast<double>(Counters_.NCurrGenerationIncomeBytesBeforeDeduplication) / Counters_.NCurrGenerationBytes;
        Counters_.CompressionRatioCharsBeforeDeduplication = static_cast<double>(Counters_.NCurrGenerationIncomeCharsBeforeDeduplication) / Counters_.NCurrGenerationChars;
    }

    void TStringCompressor::UpdateMetricsBeforeDeduplication(const TString& value, size_t count) {
        Counters_.NCurrGenerationStringsBeforeDeduplication += count;
        Counters_.NCurrGenerationIncomeBytesBeforeDeduplication += TTotalByteSize<TString>()(value) * count;
        Counters_.NCurrGenerationIncomeCharsBeforeDeduplication += value.length() * count;

        Counters_.CompressionRatioBytesBeforeDeduplication = static_cast<double>(Counters_.NCurrGenerationIncomeBytesBeforeDeduplication) / Counters_.NCurrGenerationBytes;
        Counters_.CompressionRatioCharsBeforeDeduplication = static_cast<double>(Counters_.NCurrGenerationIncomeCharsBeforeDeduplication) / Counters_.NCurrGenerationChars;
    }

    void TStringCompressor::UpdateCurrentCompressedStringsByteSize(int sign) {
        CurrentCompressedStringsByteSize_.Add((sizeof(decltype(CurrentCompressedStrings_)::value_type::value_type) + sizeof(TCompressedString::TInnerData)) * sign);
    }

    TStringCompressor::TGeneration::TGeneration(TString name, ui32 compressionLevel, ui8 generation, const TVector<TString>& trainingStrings)
        : Generation(generation)
        , StringCount(0)
        , TotalStringLength(0)
        , TotalBytes(0)
        , Name_(std::move(name))
        , Compressor_(compressionLevel)
        , SuccessfulTrain_(Train(&Compressor_, trainingStrings, Name_))
        , MemoryPool_(MemoryPoolInitialSize, TMemoryPool::TLinearGrow::Instance())
    {
        TotalBytes += Compressor_.GetDictSize();
    }

    TCompressedString::TCompressedStringInfo TStringCompressor::TGeneration::Compress(const TString& value) {
        if (SuccessfulTrain_) {
            TBuffer compressed;
            Compressor_.Compress(value, compressed);
            return SaveCompressResult(TStringBuf(compressed.Data(), compressed.size()));
        } else {
            return SaveCompressResult(TStringBuf(value));
        }
    }

    TString TStringCompressor::TGeneration::Decompress(const TCompressedString::TCompressedStringInfo& info) const {
        if (SuccessfulTrain_) {
            TBuffer decompressed;
            Compressor_.Decompress(info.GetStringBuf(), decompressed);
            TString result;
            decompressed.AsString(result);
            return result;
        } else {
            return TString(info.GetStringBuf());
        }
    }

    bool TStringCompressor::TGeneration::IsSuccessfulTrain() const {
        return SuccessfulTrain_;
    }

    bool TStringCompressor::TGeneration::Train(TZStdCompressor* compressor, const TVector<TString>& trainingStrings, const TString& name) {
        if (trainingStrings.empty()) {
            DEBUG_LOG << "trainingStrings is empty, so skipping train for TStringCompressor's TGeneration (" << name << ")" << Endl;
            return false;
        }
        try {
            compressor->Train(trainingStrings);
            return true;
        } catch (const std::exception& e) {
            ERROR_LOG << "Failed to build zstd dictionary for TStringCompressor's TGeneration (" << name << "). Will use compression without dict. Error: " << e.what() << Endl;
            return false;
        }
    }

    TCompressedString::TCompressedStringInfo TStringCompressor::TGeneration::SaveCompressResult(TStringBuf value) {
        StringCount++;

        int len = value.length();
        int lenSize = 0;
        ui8 lenBytes[8];
        while (!lenSize || len) {
            Y_ENSURE(lenSize < 8, "Length of compressed string doesn't fit in int64");
            lenBytes[7 - lenSize] = len % 256;
            len /= 256;
            lenSize++;
        }

        TCompressedString::TCompressedStringMeta meta{};
        meta.Gen = Generation;
        meta.LenSize = lenSize;

        static_assert(sizeof(meta) == 1);
        TString s;
        s.append(TStringBuf(reinterpret_cast<char*>(&meta), 1));
        s.append(TStringBuf(reinterpret_cast<char*>(lenBytes + (8 - lenSize)), lenSize));
        s.append(value);
        auto result = MemoryPool_.AppendString(TStringBuf(s));
        TotalStringLength += value.length();
        TotalBytes = MemoryPool_.MemoryAllocated();
        return TCompressedString::TCompressedStringInfo(result.Data());
    }

    void TStringCompressor::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
        ct->insert(MAKE_COUNTER_PAIR(NCurrGenerationStringsBeforeDeduplication));
        ct->insert(MAKE_COUNTER_PAIR(NCurrGenerationIncomeBytesBeforeDeduplication));
        ct->insert(MAKE_COUNTER_PAIR(NCurrGenerationIncomeCharsBeforeDeduplication));
        ct->insert(MAKE_COUNTER_PAIR(CompressionRatioBytesBeforeDeduplication));
        ct->insert(MAKE_COUNTER_PAIR(CompressionRatioCharsBeforeDeduplication));

        ct->insert(MAKE_COUNTER_PAIR(NCurrGenerationStrings));
        ct->insert(MAKE_COUNTER_PAIR(NCurrGenerationBytes));
        ct->insert(MAKE_COUNTER_PAIR(NCurrGenerationChars));
        ct->insert(MAKE_COUNTER_PAIR(NCurrGenerationIncomeBytes));
        ct->insert(MAKE_COUNTER_PAIR(NCurrGenerationIncomeChars));
        ct->insert(MAKE_COUNTER_PAIR(NPreviousGenerationBytes));
        ct->insert(MAKE_COUNTER_PAIR(NPreviousGenerationChars));

        ct->insert(MAKE_COUNTER_PAIR(NNextTrainStringsBytes));
        ct->insert(MAKE_COUNTER_PAIR(NStringsForNextTrain));
        ct->insert(MAKE_COUNTER_PAIR(NStringsForLastTrain));

        ct->insert(MAKE_COUNTER_PAIR(CompressionRatioBytes));
        ct->insert(MAKE_COUNTER_PAIR(CompressionRatioChars));

        ct->insert(MAKE_COUNTER_PAIR(RebuildIsRunning));
        ct->insert(MAKE_COUNTER_PAIR(LastRebuildTimeSpentMs));
    }
}
