#pragma once

#include "flag.h"
#include "zstd_compressor.h"

#include <travel/hotels/lib/cpp/mon/counter.h>
#include <travel/hotels/lib/cpp/mon/tools.h>

#include <travel/hotels/proto/app_config/string_compressor.pb.h>

#include <library/cpp/codecs/zstd_dict_codec.h>
#include <library/cpp/logger/global/global.h>

#include <util/datetime/cputimer.h>
#include <util/generic/deque.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/maybe.h>
#include <util/generic/string.h>
#include <util/generic/variant.h>
#include <util/generic/vector.h>
#include <util/memory/pool.h>
#include <library/cpp/deprecated/atomic/atomic.h>
#include <util/system/event.h>
#include <util/system/mutex.h>
#include <util/system/rwlock.h>
#include <util/thread/factory.h>

namespace NTravel {
    class TStringCompressorRebuildStrategy {
    public:
        TStringCompressorRebuildStrategy(double sampleProb,
                                         size_t samplesForInitialTrain,
                                         TMaybe<TDuration> rebuildEachPeriod,
                                         TMaybe<size_t> rebuildEachNSamples,
                                         TMaybe<size_t> maxTrainStringsPerRebuild);

        void Start();
        void Stop();
        void SetRebuildTrigger(std::function<void()> function);
        bool ShouldUseStringForTrain(const TString& value);
        void RebuildDone(bool successfulDictBuild);

        ~TStringCompressorRebuildStrategy();

    private:
        constexpr static double SampleProbEps_ = 1e-6;

        double SampleProb_;
        size_t SamplesForInitialTrain_;
        TMutex Mutex_;
        size_t TrainSamplesCount_;
        bool FirstTrainTriggered_;
        std::function<void()> RebuildTrigger_;
        TMaybe<TDuration> RebuidEachPeriod_;
        TMaybe<size_t> RebuildEachNSamples_;
        TMaybe<size_t> MaxTrainStringsPerRebuild_;
        THolder<IThreadFactory::IThread> RebuildTriggeringThread_;
        TAtomicFlag Stopping_;
        TManualEvent StopEvent_;

        void TriggerRebuild();
    };

    class TCompressedString {
    public:
        TCompressedString();
        const char* DangerouslyGetInnerDataOnlyForTests() const;
        bool IsDefined() const;
        size_t CalcTotalByteSize() const;
        size_t Hash() const;

        bool operator==(const TCompressedString& other) const;
        bool operator!=(const TCompressedString& other) const;

    private:
        struct TCompressedStringMeta {
            ui8 Gen : 5;
            ui8 LenSize : 3;
        };

        struct TCompressedStringInfo {
            explicit TCompressedStringInfo(const char* start)
                : Start(start)
            {
            }

            const char* Start;

            TCompressedStringMeta GetMeta() const {
                return *reinterpret_cast<const TCompressedStringMeta*>(Start);
            }

            int GetLength() const {
                auto lenSize = GetMeta().LenSize;
                size_t len = 0;
                for (size_t i = 0; i < lenSize; i++) {
                    len *= 256;
                    len |= static_cast<ui8>(*(Start + i + 1));
                }
                return len;
            }

            ui8 GetGeneration() const {
                return GetMeta().Gen;
            }

            TStringBuf GetStringBuf() const {
                auto lenSize = GetMeta().LenSize;
                return TStringBuf(reinterpret_cast<const char*>(Start + lenSize + 1), GetLength());
            }
        };

        struct TInnerData : TRefCounted<TInnerData, TAtomicCounter> {
            explicit TInnerData(TCompressedStringInfo info)
                : Info(info)
            {
            }

            std::atomic<TCompressedStringInfo> Info;
        };

        using TInnerDataPtr = TIntrusivePtr<TInnerData>;

        std::variant<TInnerDataPtr, TString> Data;

        explicit TCompressedString(const TString& value);
        explicit TCompressedString(const TInnerDataPtr& value);

        friend class TStringCompressor;
    };

    class TStringCompressor {
    public:
        TStringCompressor(TString name,
                          bool enabled,
                          ui32 compressionLevel,
                          double trainSamplingProbability,
                          size_t samplesForInitialTrainCount,
                          TMaybe<TDuration> rebuildEachPeriod = {},
                          TMaybe<size_t> rebuildEachNSampled = {},
                          TMaybe<size_t> maxTrainStringsPerRebuild = {},
                          TMaybe<TDuration> previousGenerationCleanupDelay = {});

        TStringCompressor(TString name, const NTravelProto::NAppConfig::TConfigStringCompressor& config);

        void Start();
        void Stop();
        void RegisterCounters(NMonitor::TCounterSource& source);
        TCompressedString Compress(const TString& value);
        TString Decompress(const TCompressedString& value) const;
        size_t GetStringAllocSizeOnlyForTests(const TCompressedString& value) const;
        void OnTrainFinish(std::function<void(bool)> callback);
        size_t GetStringCount() const;
        void TriggerRebuild();

        ~TStringCompressor();

    private:
        /*
         * TGeneration class is not thread safe in general.
         * But Decompress(...) implementation is thread safe if happens-before holds between Compress(...) and Decompress(...) calls.
         * This condition holds in our case because of std::atomic in TCompressedString (which holds TCompressedStringInfo).
         */
        class TGeneration {
        public:
            const ui8 Generation;
            size_t StringCount;
            size_t TotalStringLength;
            size_t TotalBytes;

            TGeneration(TString name, ui32 compressionLevel, ui8 generation, const TVector<TString>& trainingStrings);

            TCompressedString::TCompressedStringInfo Compress(const TString& value);
            TString Decompress(const TCompressedString::TCompressedStringInfo& info) const;
            bool IsSuccessfulTrain() const;

        private:
            constexpr static size_t MemoryPoolInitialSize = 16 * 1024 * 1024; // 16 MB

            const TString Name_;
            TZStdCompressor Compressor_;
            const bool SuccessfulTrain_;
            TMemoryPool MemoryPool_;

            static bool Train(TZStdCompressor* compressor, const TVector<TString>& trainingStrings, const TString& name);

            TCompressedString::TCompressedStringInfo SaveCompressResult(TStringBuf value);
        };

        struct TCounters: public NMonitor::TCounterSource {
            NMonitor::TCounter NCurrGenerationStringsBeforeDeduplication;
            NMonitor::TCounter NCurrGenerationIncomeBytesBeforeDeduplication;
            NMonitor::TCounter NCurrGenerationIncomeCharsBeforeDeduplication;
            NMonitor::TDoubleValue CompressionRatioBytesBeforeDeduplication;
            NMonitor::TDoubleValue CompressionRatioCharsBeforeDeduplication;

            NMonitor::TCounter NCurrGenerationStrings;
            NMonitor::TCounter NCurrGenerationBytes;
            NMonitor::TCounter NCurrGenerationChars;
            NMonitor::TCounter NCurrGenerationIncomeBytes;
            NMonitor::TCounter NCurrGenerationIncomeChars;

            NMonitor::TCounter NPreviousGenerationBytes;
            NMonitor::TCounter NPreviousGenerationChars;

            NMonitor::TCounter NNextTrainStringsBytes;
            NMonitor::TCounter NStringsForNextTrain;
            NMonitor::TCounter NStringsForLastTrain;

            NMonitor::TDoubleValue CompressionRatioBytes;
            NMonitor::TDoubleValue CompressionRatioChars;

            NMonitor::TCounter RebuildIsRunning;
            NMonitor::TCounter LastRebuildTimeSpentMs;

            void QueryCounters(NMonitor::TCounterTable* ct) const override;
        };

        constexpr static size_t CompressedStringsBucketCount = 65521; // prime

        const TString Name_;
        const bool Enabled_;
        const ui32 CompressionLevel_;
        std::function<void(bool)> TrainFinishCallback_;
        TDuration PreviousGenerationCleanupDelay_;

        TAtomicFlag Started_;
        TAtomicFlag Stopping_;

        TAutoEvent RebuildEvent_;
        TAutoEvent CleanupEvent_;
        THolder<IThreadFactory::IThread> RebuildThread_;

        TMutex TrainingStringsMutex_;
        TVector<TString> TrainingStrings_;

        TStringCompressorRebuildStrategy Strategy_;

        TMutex Mutex_;
        ui8 CurrentGenerationIndex_;
        TAtomicSharedPtr<TGeneration> CurrentGeneration_;
        TVector<TDeque<std::pair<size_t, TCompressedString::TInnerDataPtr>>> CurrentCompressedStrings_;
        TAtomicCounter CurrentCompressedStringsByteSize_;

        std::atomic<TGeneration*> CurrentGenerationRawPtr_;
        std::atomic<TGeneration*> PreviousGenerationRawPtr_;

        mutable TCounters Counters_;

        void StartRebuild();
        void RunRebuildThread();
        void MaybeUseStringForTrain(const TString& value);
        TMaybe<TString> DoDecompressUnlocked(const TCompressedString& value) const;
        void UpdateMetricsAfterCompress(const TString& value);
        void UpdateMetricsBeforeDeduplication(const TString& value, size_t count);
        void UpdateCurrentCompressedStringsByteSize(int sign);
    };
}

template <>
struct THash<NTravel::TCompressedString> {
    inline size_t operator()(const NTravel::TCompressedString& v) const {
        return v.Hash();
    }
};
