#include "yp_auto_resolver.h"

#include <library/cpp/logger/global/global.h>
#include <util/string/strip.h>

#include <tuple>

namespace NTravelProto {

bool operator==(const TYpConnectionInfo& firstEndpoint, const TYpConnectionInfo& secondEndpoint) {
    return std::make_tuple(firstEndpoint.GetFqdn(), firstEndpoint.GetPort()) ==
           std::make_tuple(secondEndpoint.GetFqdn(), secondEndpoint.GetPort());
}

bool operator<(const TYpConnectionInfo& firstEndpoint, const TYpConnectionInfo& secondEndpoint) {
    return std::make_tuple(firstEndpoint.GetFqdn(), firstEndpoint.GetPort()) <
           std::make_tuple(secondEndpoint.GetFqdn(), secondEndpoint.GetPort());
}

} // namespace NTravelProto

namespace NTravel {

TString ConvertToAddress(const TYpConnectionInfo& connectionInfo) {
    return connectionInfo.GetFqdn() + ":" + ToString(connectionInfo.GetPort());
}

TYpAutoResolver::TYpAutoResolver(const NTravelProto::NAppConfig::TConfigYpAutoResolver& config)
    : EndpointSetId_(config.GetEndpointSetId())
    , CooldownDuration_(TDuration::Seconds(config.GetCooldownSec()))
    , FixedAddresses_(config.GetFixedAddressList())
{
    if (!config.GetCluster().empty()) {
        TFsPath cacheDirectory = config.GetCacheDirectory();
        for (const auto& cluster : config.GetCluster()) {
            Resolvers_.push_back(MakeHolder<TYpCachedResolver>(config.GetYpServiceDiscoveryAddress(), cluster, EndpointSetId_, cacheDirectory));
        }
    }
}

TYpAutoResolver::~TYpAutoResolver() {
    Stop();
}

void TYpAutoResolver::Start() {
    INFO_LOG << "Starting YP Auto Resolver" << Endl;
    Thread_ = SystemThreadFactory()->Run(this);
}

void TYpAutoResolver::Stop() {
    INFO_LOG << "Stopping of YP Auto Resolver has begun" << Endl;
    StopFlag_.Set();
    WakeUp_.Signal();
    if (Thread_) {
        Thread_->Join();
        Thread_.Reset();
    }
    INFO_LOG << "Stopping of YP Auto Resolver has ended" << Endl;
}

void TYpAutoResolver::SetCallback(const TYpAutoResolverCallback& callback) {
    with_lock (Lock_ ) {
        Callback_ = callback;
    }
}

void TYpAutoResolver::ForceUpdate() {
    WakeUp_.Signal();
}

void TYpAutoResolver::InformSubscriber(const TVector<TYpConnectionInfo>& endpoints) const {
    TYpAutoResolverCallback callbackSnapshot;
    with_lock (Lock_) {
        callbackSnapshot = Callback_;
    }
    if (callbackSnapshot != nullptr) {
        callbackSnapshot(endpoints);
    }
}

void TYpAutoResolver::DoExecute() {
    while (!StopFlag_) {
        INFO_LOG << "Getting endpoints for " << EndpointSetId_ << Endl;
        TVector<TYpConnectionInfo> endpoints;
        for (auto& resolver : Resolvers_) {
            auto dcEndpoints = resolver->Resolve();
            endpoints.insert(endpoints.end(), dcEndpoints.begin(), dcEndpoints.end());
        }
        for (const auto& fqdn: FixedAddresses_.GetFqdn()) {
            TYpConnectionInfo i;
            i.SetPort(FixedAddresses_.GetPort());
            i.SetFqdn(fqdn);
            endpoints.push_back(i);
        }
        Sort(endpoints.begin(), endpoints.end());
        if (endpoints != LastResponse_) {
            INFO_LOG << "The list of endpoints for " << EndpointSetId_ << " has been updated" << Endl;
            InformSubscriber(endpoints);
            LastResponse_ = endpoints;
        } else {
            INFO_LOG << "The list of endpoints for " << EndpointSetId_ << " did not change" << Endl;
        }
        WakeUp_.WaitT(CooldownDuration_);
    }
}

} // namespace NTravel
