#include "yp_cached_resolver.h"

#include <infra/yp_service_discovery/api/api.pb.h>

#include <library/cpp/logger/global/global.h>

#include <google/protobuf/text_format.h>

#include <util/stream/file.h>
#include <util/string/builder.h>
#include <util/system/hostname.h>

using NTravelProto::TListOfYpConnectionInfo;

namespace NTravel {

namespace {

TString GenerateClientName() {
    return TStringBuilder() << "travel:" << HostName();
}

} // anonymous namespace

TYpCachedResolver::TYpCachedResolver(const TString& address, const TString& cluster, const TString& endpointSetId, const TFsPath& cacheDirectory)
    : Cluster_(cluster)
    , EndpointSetId_(endpointSetId)
    , CachePath_(cacheDirectory / (endpointSetId + "_" + cluster + ".pb.txt"))
    , Resolver_(GenerateClientName(), address)
    , CachedEndpoints_(LoadEndpointsFromDisk())
{
    if (!cacheDirectory.Exists()) {
        cacheDirectory.MkDirs();
    }
}

TVector<TYpConnectionInfo> TYpCachedResolver::Resolve() {
    try {
        const auto resolveResult = Resolver_.Resolve(Cluster_, EndpointSetId_);
        const TVector<NYP::NServiceDiscovery::NApi::TEndpoint> endpointList(resolveResult.endpoint_set().endpoints().begin(), resolveResult.endpoint_set().endpoints().end());
        if (endpointList.empty()) {
            ythrow yexception() << "YP Resolver must return a non-empty list of endpoints from " << Cluster_;
        }
        CachedEndpoints_ = ConvertEndpointsFromYpFormat(endpointList);
        SaveEndpointsToDisk(CachedEndpoints_);
    } catch (...) {
        ERROR_LOG << "An exception occured while resolving endpoints from " <<
            Cluster_ << ": " << CurrentExceptionMessage() << Endl;
        if (CachedEndpoints_.empty()) {
            ERROR_LOG << "Cannot recover endpoints for " << Cluster_ << Endl;
        }
    }
    return CachedEndpoints_;
}

void TYpCachedResolver::SaveEndpointsToDisk(const TVector<TYpConnectionInfo>& endpoints) const {
    try {
        TListOfYpConnectionInfo endpointList;
        *endpointList.MutableConnectionInfo() = {endpoints.begin(), endpoints.end()};
        TString output;
        if (!google::protobuf::TextFormat::PrintToString(endpointList, &output)) {
            ythrow yexception() << "Failed to serialize proto";
        }
        TOFStream(CachePath_).Write(output);
    } catch (...) {
        ERROR_LOG << "Cannot save endpoints to " << CachePath_ << ": " << CurrentExceptionMessage() << Endl;
    }
}

TVector<TYpConnectionInfo> TYpCachedResolver::LoadEndpointsFromDisk() const {
    try {
        TListOfYpConnectionInfo endpointList;
        if (!google::protobuf::TextFormat::ParseFromString(TIFStream(CachePath_).ReadAll(), &endpointList)) {
            ythrow yexception() << "Failed to parse proto from " << CachePath_;
        }
        if (endpointList.GetConnectionInfo().empty()) {
            ythrow yexception() << "Got an empty list of endpoints from " << CachePath_;
        }
        INFO_LOG << "Successfully loaded endpoints from " << CachePath_ << Endl;
        return {endpointList.GetConnectionInfo().begin(), endpointList.GetConnectionInfo().end()};
    } catch (...) {
        ERROR_LOG << "Cannot load endpoints from " << CachePath_ << ": " << CurrentExceptionMessage() << Endl;
        return {};
    }
}

TVector<TYpConnectionInfo> TYpCachedResolver::ConvertEndpointsFromYpFormat(
    const TVector<NYP::NServiceDiscovery::NApi::TEndpoint>& endpoints)
{
    TVector<TYpConnectionInfo> connectionInfos;
    connectionInfos.reserve(endpoints.size());
    for (const auto& endpoint : endpoints) {
        TYpConnectionInfo connectionInfo;
        connectionInfo.SetFqdn(endpoint.fqdn());
        connectionInfo.SetPort(endpoint.port());
        connectionInfos.push_back(connectionInfo);
    }
    return connectionInfos;
}

} // namespace NTravel
