#pragma once

#include "tools.h"
#include "data.h"
#include "queue_reader_cache.h"
#include "deduplicaton_service.h"

#include <travel/proto/commons.pb.h>
#include <travel/hotels/proto2/hotels.pb.h>
#include <travel/hotels/proto/app_config/yt_key_value_storage.pb.h>

#include <travel/hotels/lib/cpp/mon/counter.h>
#include <travel/hotels/lib/cpp/mon/counter_hypercube.h>
#include <travel/hotels/lib/cpp/mon/tools.h>
#include <travel/hotels/lib/cpp/util/flag.h>
#include <travel/hotels/lib/cpp/util/profiletimer.h>
#include <travel/hotels/lib/cpp/util/hash_set_with_ttl.h>

#include <yt/yt/client/api/client.h>

#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/map.h>
#include <util/generic/vector.h>
#include <util/generic/queue.h>
#include <library/cpp/deprecated/atomic/atomic.h>
#include <util/system/mutex.h>
#include <util/system/event.h>
#include <util/thread/factory.h>
#include <util/thread/pool.h>

namespace NTravel {
class TYtKeyValueStorage {
public:
    struct TRecord {
        TInstant Timestamp;
        TInstant ExpireTimestamp;
        TString  Bytes;
        TString  MessageId;
        TString  MessageType;
    };

    TYtKeyValueStorage(const NTravelProto::NAppConfig::TConfigYtKeyValueStorage& config, const TString& name);
    ~TYtKeyValueStorage();

    void RegisterCounters(NMonitor::TCounterSource& source);

    bool IsReady() const;
    void Start();
    void Stop();

    template<class TProtoMessage>
    TMaybe<TProtoMessage> Read(const TString& key) const {
        auto record = DoRead({key}).at(0);
        if (record.Empty()) {
            return {};
        }
        TProtoMessage message;
        if (message.GetDescriptor()->full_name() != record.GetRef().MessageType) {
            throw yexception() << "Unexpected message type: expected '" << message.GetDescriptor()->full_name() << "', got '" << record.GetRef().MessageType << "'";
        }
        if (!message.ParseFromString(record.GetRef().Bytes)) {
            throw yexception() << "Failed to parse proto message";
        }
        return message;
    }
private:
    struct TCounters : public NMonitor::TCounterSource {
        TCounters();

        NMonitor::TCounter          IsReady;

        NMonitor::TDerivCounter     NRequests;
        NMonitor::TDerivCounter     NTotalKeysRequested;
        NMonitor::TDerivCounter     NNotFound;
        NMonitor::TDerivCounter     NFound;
        NMonitor::TDerivCounter     NReadError;
        NMonitor::TDerivHistogramCounter ReadTimeMs;

        void QueryCounters(NMonitor::TCounterTable* ct) const override;
    };

    struct TPerClusterCounters : public NMonitor::TCounterSource {
        TPerClusterCounters();

        NMonitor::TCounter          IsReady;

        NMonitor::TDerivCounter     NReadError;
        NMonitor::TDerivHistogramCounter ReadTimeMs;

        void QueryCounters(NMonitor::TCounterTable* ct) const override;
    };

    struct TCluster {
        TString Name;
        NYT::NApi::IClientPtr YtClient;
    };

    enum EReadStatus {
        Unknown,
        Failed,
        NotFound,
        Found,
    };

    const TDuration LivenessCheckTimeout_ = TDuration::Seconds(3);
    const TDuration CheckLivenessEachPeriod_ = TDuration::Seconds(10);
    const NTravelProto::NAppConfig::TConfigYtKeyValueStorage Config_;
    const TString Name_;

    NTravel::NYtRpc::TClientCreator ClientCreator_;

    TMutex ClustersMutex_;
    THashMap<TString, TCluster> Clusters_;

    TAtomicFlag IsReady_;
    TAtomicFlag Stopping_;
    TManualEvent StopEvent_;
    THolder<IThreadFactory::IThread> LivenessCheckThread_;

    const TString TablePath_;
    const size_t MinReadsForSuccess_;
    const size_t MinReadsForNotFound_;
    const TDuration ReadTimeout_;

    mutable TCounters Counters_;
    mutable NMonitor::TCounterHypercube<TPerClusterCounters> PerClusterCounters_;

    void RunLivenessChecking();

    NYT::TSharedRange<NYT::NTableClient::TLegacyKey> PrepareRange(const TVector<TString>& keys, i32 keyField) const;
    TVector<NYT::TFuture<NYT::NApi::IUnversionedRowsetPtr>> SendYtLookupRequests(const NYT::NTableClient::TNameTablePtr& nameTable,
                                                                                 const NYT::TSharedRange<NYT::NTableClient::TLegacyKey>& keys) const;
    std::pair<EReadStatus, TVector<NYT::TFuture<NYT::NApi::IUnversionedRowsetPtr>> /* futuresToWait */> ProcessReadFutures(
        const TVector<NYT::TFuture<NYT::NApi::IUnversionedRowsetPtr>>& readFutures,
        size_t rowsCnt) const;

    TVector<TMaybe<TRecord>> DoRead(const TVector<TString>& keys) const;
};

}// namespace NTravel
