#include "lock.h"

#include <library/cpp/logger/global/global.h>

#include <travel/hotels/lib/cpp/util/secret_reader.h>

#include <util/system/hostname.h>

namespace NTravel {

TYtLock::TYtLock(const NTravelProto::NAppConfig::TConfigYtLock& cfg)
    : Config_(cfg)
    , IsFallbackHost_(Config_.GetFallbackHost() == FQDNHostName())
{
    YtClient_ = NYT::CreateClient(Config_.GetYtProxy(), NYT::TCreateClientOptions().Token(ReadSecret(Config_.GetYtTokenPath())));
}

TYtLock::~TYtLock() {
    Stop();
}

void TYtLock::SetStartHandler(std::function<void()> onStart) {
    OnStart_ = onStart;
}

void TYtLock::SetStopHandler(std::function<void()> onStop) {
    OnStop_ = onStop;
}

void TYtLock::Stop() {
    IsRunning_.Clear();
    WakeUp_.Signal();
}

void TYtLock::OnStart() {
    if (!OnStart_) {
        return;
    }
    INFO_LOG << "Calling start handler" << Endl;
    OnStart_();
}

void TYtLock::OnStop() {
    if (!OnStop_) {
        return;
    }
    INFO_LOG << "Calling stop handler" << Endl;
    OnStop_();
}

TYtLock::ELockStatus TYtLock::TryAcquireLock(const std::function<void()>& beforeLock, const std::function<void()>& afterLock) {
    try {
        INFO_LOG << "Trying to acquire lock on node " << Config_.GetNodePath() << Endl;

        if (!YtClient_->Exists(Config_.GetNodePath())) {
            WARNING_LOG << "Node " << Config_.GetNodePath() << " does not exist, trying to create" << Endl;
            YtClient_->Create(Config_.GetNodePath(), NYT::NT_DOCUMENT, NYT::TCreateOptions().Recursive(true));
        }

        auto tx = YtClient_->StartTransaction(NYT::TStartTransactionOptions()
                                              .Timeout(TDuration::Seconds(Config_.GetTxTimeoutSec()))
                                              .AutoPingable(false)
                                              .Title("Lock at host '" + FQDNHostName() + "'")
        );
        auto lock = tx->Lock(Config_.GetNodePath(), NYT::ELockMode::LM_EXCLUSIVE, NYT::TLockOptions().Waitable(true));
        try {
            lock->Wait(TDuration::Seconds(Config_.GetPingPeriodSec()));
        } catch (const NThreading::TFutureException& ex) {
            WARNING_LOG << "Lock on node " << Config_.GetNodePath() << " was already acquired by someone else ('"
                        << ex.what() << "')" << Endl;
            return ELockStatus::TimedOut;
        }

        INFO_LOG << "Lock with Id = " << GetGuidAsString(lock->GetId()) << " was successfully acquired" << Endl;
        auto pingFunc = [tx, lock]() {
            try {
                tx->Ping();
                return true;
            } catch (...) {
                ERROR_LOG << "Failed to ping transaction with Id = " << GetGuidAsString(lock->GetId())
                          << ": " << CurrentExceptionMessage() << Endl;
                return false;
            }
        };
        if (beforeLock) {
            beforeLock();
        }
        DoPeriodically(pingFunc);
        if (afterLock) {
            afterLock();
        }
    } catch (...) {
        ERROR_LOG << "Failed to acquire lock: " << CurrentExceptionMessage() << Endl;
        return ELockStatus::Failed;
    }

    return ELockStatus::Acquired;
}

void TYtLock::Run() {
    IsRunning_.Set();
    std::function<void()> beforeLock = [this] { OnStart(); };
    std::function<void()> afterLock = [this] { OnStop(); };
    ui32 failedTries = 0;
    bool inFallbackMode = false;
    while (IsRunning_) {
        switch (TryAcquireLock(beforeLock, afterLock)) {
            case ELockStatus::Failed:
                if (!IsFallbackHost_ || inFallbackMode) {
                    break;
                }
                ++failedTries;
                if (failedTries < Config_.GetNumTriesBeforeFallback()) {
                    break;
                }
                ERROR_LOG << "Cannot acquire lock after several tries, falling back to this host" << Endl;
                inFallbackMode = true;
                beforeLock();
                beforeLock = {};
                break;
            case ELockStatus::TimedOut:
                if (inFallbackMode) {
                    afterLock();
                    return;
                }
                failedTries = 0;
                break;
            case ELockStatus::Acquired:
                return;
        }
        WakeUp_.WaitT(TDuration::Seconds(Config_.GetRetryTimeoutSec()));
    }
}

void TYtLock::DoPeriodically(std::function<bool()> periodicFunc) {
    while (IsRunning_) {
        if (!periodicFunc()) {
            return;
        }
        WakeUp_.WaitT(TDuration::Seconds(Config_.GetPingPeriodSec()));
    }
}

}  // namespace NTravel
