#pragma once

#include <travel/hotels/lib/cpp/yt/table_cache.h>
#include <travel/hotels/lib/cpp/yt/tools.h>
#include <travel/hotels/lib/cpp/mon/tools.h>
#include <travel/hotels/lib/cpp/mon/counter.h>
#include <travel/hotels/lib/cpp/util/profiletimer.h>
#include <travel/hotels/lib/cpp/protobuf/tools.h>

#include <mapreduce/yt/interface/node.h>

#include <library/cpp/logger/global/global.h>

#include <util/generic/ptr.h>
#include <util/generic/hash.h>
#include <util/system/rwlock.h>

#include <utility>
#include <functional>
#include <type_traits>
#include <memory>

namespace NTravel {

template <class TTKey, class TTProtoRecord>
class TYtPersistentConfig {
public:
    using TKey = TTKey;
    using TProtoRecord = TTProtoRecord;
    using TMapping = THashMap<TKey, TProtoRecord>;
    using TOnUpdateFunc = std::function<void (bool first)>;

    TYtPersistentConfig(const TString& name, const NTravelProto::NAppConfig::TYtTableCacheConfig& config)
        : Name_(name)
        , TableCache_(name, config)
        , Mapping_(std::make_shared<TMapping>())
        , NewMapping_(std::make_shared<TMapping>())
    {
        if (!TableCache_.IsEnabled()) {
            throw yexception() << "Cannot work with not enabled " << name;
        }
        TableCache_.SetCallbacks(this, &TYtPersistentConfig::OnConvert, &TYtPersistentConfig::OnData, &TYtPersistentConfig::OnFinish);
    }

    virtual ~TYtPersistentConfig() {
        Stop();
    }

    void Start() {
        TableCache_.Start();
    }

    void Stop() {
        TableCache_.Stop();
    }

    bool IsReady() const {
        return TableCache_.IsReady();
    }

    void RegisterCounters(NMonitor::TCounterSource& source) {
        TableCache_.RegisterCounters(source, Name_);
    }

    std::shared_ptr<const TProtoRecord> GetById(const TKey& key) const {
        auto mapping = GetAll();
        auto it = mapping->find(key);
        if (it == mapping->end()) {
            return {};
        }
        return {mapping, &it->second};
    }

    std::shared_ptr<const TMapping> GetAll() const {
        TReadGuard g(Lock_);
        return Mapping_;
    }

    size_t Size() const {
        return GetAll()->size();
    }

    void SetOnUpdateHandler(TOnUpdateFunc func) {
        OnUpdateFunc_ = std::move(func);
    }

private:
    const TString Name_;

    TYtTableCache<TProtoRecord> TableCache_;

    TOnUpdateFunc OnUpdateFunc_;

    TRWMutex Lock_;
    std::shared_ptr<TMapping> Mapping_;
    std::shared_ptr<TMapping> NewMapping_;

    virtual void OnConvert(const NYT::TNode& node, TProtoRecord* proto) const {
        NTravel::NProtobuf::NodeToProto(node, proto);
    }

    virtual void OnData(const TProtoRecord& proto) {
        auto res = NewMapping_->emplace(NTravel::NProtobuf::GetIdentifier<TKey, TProtoRecord>(proto), proto);
        if (!res.second) {
            INFO_LOG << "Merging for record " << res.first->first << Endl;
            res.first->second.MergeFrom(proto);
        }
    }

    virtual void OnFinish(bool ok, bool initial) {
        if (ok) {
            {
                TWriteGuard m(Lock_);
                Mapping_.swap(NewMapping_);
            }
            if (initial) {
                INFO_LOG << "Got new " << Name_ << " config" << Endl;
            } else {
                INFO_LOG << "Got initial " << Name_ << " config" << Endl;
            }
            if (OnUpdateFunc_) {
                OnUpdateFunc_(initial);
            }
        }
        NewMapping_ = std::make_shared<TMapping>();
    }
};

}// namespace NTravel
