#pragma once

#include <travel/hotels/proto/app_config/yt_table_cache.pb.h>

#include <travel/hotels/lib/cpp/yt/table_cache.h>
#include <travel/hotels/lib/cpp/mon/tools.h>
#include <travel/hotels/lib/cpp/mon/counter.h>
#include <travel/hotels/lib/cpp/util/profiletimer.h>

#include <mapreduce/yt/interface/node.h>

#include <library/cpp/logger/global/global.h>

#include <util/generic/algorithm.h>
#include <util/generic/ptr.h>
#include <util/generic/hash.h>
#include <util/generic/deque.h>
#include <util/generic/maybe.h>
#include <util/system/rwlock.h>

#include <utility>
#include <functional>
#include <type_traits>
#include <memory>

namespace NTravel {

template <typename TTKey, typename TTValue, typename TTProtoRecord>
class TYtPersistentMapper {
public:
    using TKey = TTKey;
    using TValue = TTValue;
    using TProtoRecord = TTProtoRecord;

    using TMappingElement = std::pair<TKey, TValue>;
    using TMapping = TDeque<TMappingElement>;

    struct TCounters: public NMonitor::TCounterSource {
        NMonitor::TCounter NBytes;
        NMonitor::TCounter NElements;

        void QueryCounters(NMonitor::TCounterTable* ct) const override {
            ct->insert(MAKE_COUNTER_PAIR(NBytes));
            ct->insert(MAKE_COUNTER_PAIR(NElements));
        }
    };

    TYtPersistentMapper(const TString& name, const NTravelProto::NAppConfig::TYtTableCacheConfig& config)
        : Name_(name)
        , NewMapping_(std::make_shared<TMapping>())
        , Cache_(name, config)
        , Mapping_(std::make_shared<TMapping>())
    {
        Cache_.SetCallbacks(this, &TYtPersistentMapper::OnConvert, &TYtPersistentMapper::OnData, &TYtPersistentMapper::OnFinish);
    }

    virtual ~TYtPersistentMapper() {
        Stop();
    }

    void Start() {
        Cache_.Start();
    }

    void Stop() {
        Cache_.Stop();
    }

    bool IsReady() const {
        return Cache_.IsReady();
    }

    std::shared_ptr<const TValue> GetMapping(const TKey& key) const {
        auto mapping = GetMapping();
        auto it = LowerBoundBy(mapping->cbegin(), mapping->cend(), key, TYtPersistentMapper::GetKey);
        if ((it == mapping->end()) || (key < GetKey(*it))) {
            return {};
        }
        return {mapping, &it->second};
    }

    void SetMappingFilters(std::function<void(TKey* key)> keyMapper, std::function<void(TValue* to, TValue* from)> valueJoiner) {
        Y_ASSERT(keyMapper || !valueJoiner);
        auto g = Guard(CallbacksLock_);
        KeyMapper_ = keyMapper;
        ValueJoiner_ = valueJoiner;
    }

    void SetOnFinishHandler(std::function<void()> onFinishHandler) {
        auto g = Guard(CallbacksLock_);
        OnFinishHandler_ = onFinishHandler;
    }

    void Reload() {
        Cache_.Reload();
    }

    void RegisterCounters(NMonitor::TCounterSource& source) {
        source.RegisterSource(&Counters_, Name_);
        Cache_.RegisterCounters(source, Name_ + "Cache");
    }
protected:
    const TString Name_;
    mutable TCounters Counters_;
    std::shared_ptr<TMapping> NewMapping_;

    std::shared_ptr<const TMapping> GetMapping() const {
        TReadGuard g(MappingLock_);
        return Mapping_;
    }

    virtual size_t GetAllocSize(const TMappingElement& element) const {
        Y_UNUSED(element);
        return 0;
    }

    virtual void OnConvert(const NYT::TNode& node, TProtoRecord* proto) const = 0;
    virtual void OnData(const TProtoRecord& proto) = 0;
private:
    TYtTableCache<TProtoRecord> Cache_;

    TRWMutex MappingLock_;
    std::shared_ptr<TMapping> Mapping_;

    TMutex CallbacksLock_;
    std::function<void(TKey* key)> KeyMapper_;
    std::function<void(TValue* to, TValue* from)> ValueJoiner_;
    std::function<void()> OnFinishHandler_;

    static const TKey& GetKey(const TMappingElement& element) {
        return element.first;
    }

    static bool MappingElementLess(const TMappingElement& lhs, const TMappingElement& rhs) {
        return GetKey(lhs) < GetKey(rhs);
    }

    void OnFinish(bool ok, bool /*initial*/) {
        if (ok) {
            if (!NewMapping_->empty()) {
                size_t mappingOldSize = NewMapping_->size();
                TProfileTimer started;
                {
                    auto g = Guard(CallbacksLock_);

                    if (KeyMapper_) {
                        for (auto& keyValue: *NewMapping_) {
                            KeyMapper_(&keyValue.first);
                        }
                    }

                    Sort(*NewMapping_, TYtPersistentMapper::MappingElementLess);

                    auto first = std::begin(*NewMapping_);
                    const auto last = std::end(*NewMapping_);
                    auto curr = first;
                    while (++curr != last) {
                        if (!MappingElementLess(*first, *curr)) {
                            if (ValueJoiner_) {
                                ValueJoiner_(&first->second, &curr->second);
                            }
                        } else if (++first != curr) {
                            *first = std::move(*curr);
                        }
                    }
                    NewMapping_->erase(std::next(first), last);
                    NewMapping_->shrink_to_fit();
                    Y_ASSERT(IsSorted(std::cbegin(*NewMapping_), std::cend(*NewMapping_), TYtPersistentMapper::MappingElementLess));
                }
                DEBUG_LOG << Name_ << ": Mapping remapped in " << started.Get() << Endl;
                if (mappingOldSize > NewMapping_->size()) {
                    INFO_LOG << Name_ << ": " << (mappingOldSize - NewMapping_->size()) << " of " << mappingOldSize << " values are merged due to non-uniqueness" << Endl;
                }
            }
            size_t allocSize = 0;
            for (const auto& mappingElement: *NewMapping_) {
                allocSize += sizeof mappingElement + GetAllocSize(mappingElement);
            }
            Counters_.NBytes = allocSize;
            Counters_.NElements = NewMapping_->size();
            {
                TWriteGuard g(MappingLock_);
                Mapping_ = std::move(NewMapping_);
            }
            {
                auto g = Guard(CallbacksLock_);
                if (OnFinishHandler_) {
                    OnFinishHandler_();
                }
            }
        }
        NewMapping_ = std::make_shared<TMapping>();
    }
};

}// namespace NTravel
