#pragma once

#include "tools.h"
#include "data.h"
#include "queue_reader_cache.h"
#include "deduplicaton_service.h"

#include <travel/hotels/proto2/hotels.pb.h>
#include <travel/hotels/proto/app_config/yt_queue.pb.h>

#include <travel/hotels/lib/cpp/mon/counter.h>
#include <travel/hotels/lib/cpp/mon/counter_hypercube.h>
#include <travel/hotels/lib/cpp/mon/tools.h>
#include <travel/hotels/lib/cpp/util/flag.h>
#include <travel/hotels/lib/cpp/util/profiletimer.h>
#include <travel/hotels/lib/cpp/util/hash_set_with_ttl.h>

#include <yt/yt/client/api/client.h>

#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/map.h>
#include <util/generic/vector.h>
#include <library/cpp/deprecated/atomic/atomic.h>
#include <util/system/mutex.h>
#include <util/system/event.h>
#include <util/thread/factory.h>
#include <util/thread/pool.h>

namespace NTravel {

struct TYtQueueReaderOptions {
    bool EnableDeduplication = true;
    bool CheckExpireTimestamp = true;

    TYtQueueReaderOptions& SetEnableDeduplication(bool val) {
        EnableDeduplication = val;
        return *this;
    }
    TYtQueueReaderOptions& SetCheckExpireTimestamp(bool val) {
        CheckExpireTimestamp = val;
        return *this;
    }
};

class TYtQueueReader {
public:
    using TRecordConverter = std::function<void(TYtQueueMessage& message)>;
    using TRecordHandler = std::function<bool(const TYtQueueMessage& message)>;
    using TReadinessNotifier = std::function<void(void)>;

    TYtQueueReader(const NTravelProto::NAppConfig::TConfigYtQueueReader& config, const TString& name, const TYtQueueReaderOptions& opts = {});
    ~TYtQueueReader();

    void RegisterCounters(NMonitor::TCounterSource& source);
    bool IsReady() const;

    void Start();
    void Stop();

    void Ignore(const NProtoBuf::Message& message);

    void Subscribe(const NProtoBuf::Message& message, TRecordHandler handler);
    template <class TOwner>
    void Subscribe(const NProtoBuf::Message& message, TOwner* owner, bool (TOwner::*handler)(const TYtQueueMessage& message)) {
        Subscribe(message, [owner, handler](const TYtQueueMessage& message) {
            return (owner->*handler)(message);
        });
    }

    void RegisterConverter(const NProtoBuf::Message& message, TRecordConverter converter);
    template <class TOwner>
    void RegisterConverter(const NProtoBuf::Message& message, TOwner* owner, void (TOwner::*converter)(TYtQueueMessage& message)) {
        RegisterConverter(message,
            [owner, converter](TYtQueueMessage& message) {
              (owner->*converter)(message);
            }
        );
    }

    void SetReadinessNotifier(const TReadinessNotifier& notifier);

    void TrimTable(const TYtQueueMessageOrigin& origin);
private:
    struct TCounters : public NMonitor::TCounterSource {
        NMonitor::TCounter      IsReady;
        NMonitor::TCounter      NAliveClusters;
        NMonitor::TDerivCounter NRecordsGood;
        NMonitor::TDerivCounter NRecordsBad;
        NMonitor::TDerivCounter NRecordsUnknown;
        NMonitor::TDerivCounter NRecordsDeduplicated;
        NMonitor::TDerivCounter NRecordsNotConsumed;
        NMonitor::TDerivCounter NInvalidTrims;

        NMonitor::TDerivCounter         NReadError; // Единичные ошибки чтения
        mutable NMonitor::TCounter      LongNoReadError;// Давно ничего вообще не удаётся читать
        NMonitor::TDerivCounter         MsgUnpackNs;
        mutable NMonitor::TCounter      NProcessQueueSize;

        TYtQueueReader&           Owner;

        TCounters(TYtQueueReader& owner);
        void QueryCounters(NMonitor::TCounterTable* ct) const override;
    };

    struct TPerClusterCounters : public NMonitor::TCounterSource {
        NMonitor::TTimeValue LastProcessedTimestamp;
        NMonitor::TCounter ActualizationLagSec;

        void QueryCounters(NMonitor::TCounterTable* ct) const override;
    };

    const NTravelProto::NAppConfig::TConfigYtQueueReader Config_;
    const i64 ReadLimit_;
    const TString Name_;
    const TYtQueueReaderOptions Options_;
    TDuration RecordMaxAge_;

    NTravel::NYtRpc::TClientCreator ClientCreator_;

    mutable TCounters Counters_;
    mutable NMonitor::TCounterHypercube<TPerClusterCounters> PerClusterCounters_;

    TAutoPtr<IThreadFactory::IThread> StartThread_;
    TAtomicFlag StopFlag_;

    struct TCluster {
        TString Name;
        TString PrintName;

        TAutoPtr<IThreadFactory::IThread> Thread;
        TAutoEvent WakeUp;

        NYT::NApi::IClientPtr YtClient;
        bool IsAlive = false;

        i64 CurrentRow = -1;
    };

    TThreadPool ProcessThreads_;

    THashMap<TString, TCluster> Clusters_;
    size_t InitialReadClusterCount_;

    TAtomicFlag IsReady_;
    TAtomic InitialReadDoneCount_;

    TMutex LastSuccessfulReadLock_;
    TProfileTimer LastSuccessfulRead_;

    TDeduplicationService DeduplicationService_;

    THashMap<TString, TRecordHandler> Handlers_;

    THashMap<TString, TRecordConverter> Converters_;

    TReadinessNotifier ReadinessNotifier_;

    TYtQueueReaderCache Cache_;

    void SetIsReady();

    bool IsLongNoReadError() const;

    void StartThreadFunc();

    void ThreadFunc(TCluster& cluster);

    void ChangeClusterAlive(TCluster& cluster, bool isAlive);

    void InitialCacheRead();

    bool ReadTable(TCluster& cluster, int tabletIndex);

    TDuration DetermineRecordMaxAge();
    NYT::NApi::TTabletInfo GetTabletInfo(const TCluster& cluster, int tabletIndex) const;

    i64 FindApproximateLowerBoundByTimestamp(TCluster& cluster, int tabletIndex, TInstant timestamp) const;

    TMaybe<TInstant> GetRecordTimestamp(TCluster& cluster, int tabletIndex, i64 rowIndex) const;

    // returns last read row index, or -1 if none
    i64 ReadTable(const TCluster& cluster, int tabletIndex, i64 rowFrom, i64 rowTo);

    void UnpackMessage(const TYtQueueMessagePacked& msg, TYtQueueMessage& msgUnpacked) const;
    void PackMessage(const TYtQueueMessage& msgUnpacked, TYtQueueMessagePacked& msg) const;

    TString Decompress(NTravelProto::EMessageCodec codec, TString bytes) const;

    void MaybeInitCurrentRowFromYt(TCluster& cluster, int tabletIndex);
};

}// namespace NTravel
