#pragma once

#include "data.h"

#include <travel/hotels/proto/app_config/yt_queue.pb.h>
#include <travel/hotels/lib/cpp/util/flag.h>
#include <travel/hotels/lib/cpp/sqlite/db.h>
#include <travel/hotels/lib/cpp/mon/tools.h>

#include <util/datetime/base.h>
#include <util/generic/noncopyable.h>
#include <util/system/event.h>
#include <util/generic/vector.h>
#include <util/generic/deque.h>
#include <util/generic/ptr.h>
#include <util/system/mutex.h>
#include <util/thread/factory.h>

namespace NTravel {

class TYtQueueReaderCache: TNonCopyable {
public:
    TYtQueueReaderCache(const NTravelProto::NAppConfig::TConfigYtQueueReader& config);
    ~TYtQueueReaderCache();

    void RegisterCounters(NMonitor::TCounterSource& source, const TString& name);

    void Start();
    void Stop();

    void DeleteOldRows(TInstant now, TDuration maxAge, bool checkExpireTimestamp);
    void Trim(const TYtQueueMessageOrigin& till);
    void ReadRows(TInstant since, std::function<bool(const TYtQueueMessagePacked& message)> cb);
    void ReadIndicies(std::function<void(const TYtQueueMessageOrigin& origin)> cb);
    void Write(TVector<TYtQueueMessagePacked>&& messages, const TYtQueueMessageOrigin& origin);
private:
    struct TCounters : public NMonitor::TCounterSource {
        NMonitor::TCounter NCacheBroken;
        NMonitor::TCounter NWriteQueueMessages;
        NMonitor::TDerivCounter NTransactions;
        NMonitor::TDerivCounter NReadItems;

        void QueryCounters(NMonitor::TCounterTable* ct) const override;
    };

    using TOriginKey = std::pair<TString/*Cluster*/, int/*TabletIndex*/>;

    struct TWriteItem : public TThrRefBase   {
        TVector<TYtQueueMessagePacked> Messages;
        THashMap<TOriginKey, i64/*RowIndex*/> Indecies;
    };
    using TWriteItemRef = TIntrusivePtr<TWriteItem>;


    const TString LogPrefix_;
    const size_t MaxWriteQueueItems_;
    const size_t MinBatchSize_;
    const size_t MaxBatchSize_;
    const i64    DataVersion_;

    TCounters Counters_;
    TAtomicFlag Working_;

    TMutex DbLock_;
    THolder<NSQLite::TDatabase> Db_;

    TAutoEvent WakeUpEvent_;
    TAtomicFlag StopFlag_;

    TMutex WriteLock_;
    TWriteItemRef WriteItem_;
    TDeque<TWriteItemRef> WriteQueue_;
    TAutoPtr<IThreadFactory::IThread> WriteThread_;

    void WithTransaction(const TString& name, std::function<void(NSQLite::TTransaction&)> func);
    void WriteThreadLoop();
    void OnCacheBroken();
    void InitDb(NSQLite::TDatabase* db);
    int GetDbVersion(NSQLite::TDatabase* db);
    void CheckDataVersion();
};

}// namespace NTravel
