#include "queue_reader_with_confirmation.h"

#include <travel/hotels/lib/cpp/scheduler/scheduler.h>

namespace NTravel {

TYtQueueReaderWithConfirmation::TRowIndexWithStatus::TRowIndexWithStatus(i64 rowIndex, bool isCompleted)
    : RowIndex(rowIndex)
    , IsCompleted(isCompleted)
{
}

bool TYtQueueReaderWithConfirmation::TRowIndexWithStatus::operator<(const TRowIndexWithStatus& rhs) const {
    return std::make_tuple(RowIndex, IsCompleted) < std::make_tuple(rhs.RowIndex, rhs.IsCompleted);
}

TYtQueueReaderWithConfirmation::TYtQueueReaderWithConfirmation(const NTravelProto::NAppConfig::TConfigYtQueueReader& config, const TString& name, TDuration maxAge)
    : MaxAge_(maxAge)
    , Reader_(config, name, TYtQueueReaderOptions().SetEnableDeduplication(false))
{
}

void TYtQueueReaderWithConfirmation::RegisterCounters(NMonitor::TCounterSource& source) {
    Reader_.RegisterCounters(source);
}

void TYtQueueReaderWithConfirmation::Start() {
    NTravel::TScheduler::Instance().EnqueuePeriodical(MaxAge_, [this]() {
        with_lock (Lock_) {
            InProgressRecordOriginsByRecordIdFirstGen_.swap(InProgressRecordOriginsByRecordIdSecondGen_);
            InProgressRecordOriginsByRecordIdSecondGen_.clear();
            ActiveRowIndicesByTabletOriginFirstGen_.swap(ActiveRowIndicesByTabletOriginSecondGen_);
            ActiveRowIndicesByTabletOriginSecondGen_.clear();
            CompletedRecordsFirstGen_.swap(CompletedRecordsSecondGen_);
            CompletedRecordsSecondGen_.clear();
        }
    });
    Reader_.Start();
}

void TYtQueueReaderWithConfirmation::Stop() {
    Reader_.Stop();
}

void TYtQueueReaderWithConfirmation::Subscribe(const NProtoBuf::Message& message, const TConfRecordHandler& handler) {
    Reader_.Subscribe(message, [this, handler] (const TYtQueueMessage& message) {
        TYtQueueMessageOrigin partialOrigin(message.Origin);
        partialOrigin.RowIndex = 0;
        const auto& recordId = message.MessageId;
        bool firstRecord = false;
        with_lock (Lock_) {
            if (CompletedRecordsSecondGen_.contains(recordId) || CompletedRecordsFirstGen_.contains(recordId)) {
                MarkRecordOriginAsCompletedUnlocked(message.Origin);
            } else {
                firstRecord = !InProgressRecordOriginsByRecordIdSecondGen_.contains(recordId) &&
                              !InProgressRecordOriginsByRecordIdFirstGen_.contains(recordId);
                InProgressRecordOriginsByRecordIdSecondGen_[recordId].push_back(message.Origin);
                ActiveRowIndicesByTabletOriginSecondGen_[partialOrigin].emplace(message.Origin.RowIndex, false);
            }
        }

        if (firstRecord && handler) {
            return handler(message);
        }
        return true;
    });
}

void TYtQueueReaderWithConfirmation::MarkRecordAsCompleted(const TString& recordId) {
    with_lock (Lock_) {
        for (auto* inProgressRecordOriginsByRecordId : {&InProgressRecordOriginsByRecordIdFirstGen_, &InProgressRecordOriginsByRecordIdSecondGen_}) {
            for (const auto& recordOrigin : (*inProgressRecordOriginsByRecordId)[recordId]) {
                MarkRecordOriginAsCompletedUnlocked(recordOrigin);
            }
        }
        CompletedRecordsSecondGen_.insert(recordId);
    }
}

void TYtQueueReaderWithConfirmation::MarkRecordOriginAsCompletedUnlocked(const TYtQueueMessageOrigin& recordOrigin) {
    TYtQueueMessageOrigin partialOrigin(recordOrigin);
    partialOrigin.RowIndex = 0;
    auto& firstGen = ActiveRowIndicesByTabletOriginFirstGen_[partialOrigin];
    auto& secondGen = ActiveRowIndicesByTabletOriginSecondGen_[partialOrigin];

    firstGen.erase({recordOrigin.RowIndex, false});
    secondGen.erase({recordOrigin.RowIndex, false});
    secondGen.emplace(recordOrigin.RowIndex, true);

    i64 trimRowIndex = -1;
    bool somethingChanged = true;
    while (somethingChanged) {
        somethingChanged = false;
        if (!firstGen.empty() && firstGen.begin()->IsCompleted && (secondGen.empty() || firstGen.begin()->RowIndex < secondGen.begin()->RowIndex)) {
            somethingChanged = true;
            trimRowIndex = firstGen.begin()->RowIndex;
            firstGen.erase(firstGen.begin());
        }
        if (!secondGen.empty() && secondGen.begin()->IsCompleted && (firstGen.empty() || secondGen.begin()->RowIndex < firstGen.begin()->RowIndex)) {
            somethingChanged = true;
            trimRowIndex = secondGen.begin()->RowIndex;
            secondGen.erase(secondGen.begin());
        }
    }

    if (trimRowIndex != -1) {
        partialOrigin.RowIndex = trimRowIndex;
        Reader_.TrimTable(partialOrigin);
    }
}

} // namespace NTravel
