#pragma once

#include <travel/hotels/proto/app_config/yt_table_cache.pb.h>
#include <travel/hotels/lib/cpp/yt/table_cache.pb.h>
#include <travel/hotels/lib/cpp/util/flag.h>
#include <travel/hotels/lib/cpp/mon/counter.h>
#include <travel/hotels/lib/cpp/mon/tools.h>
#include <travel/hotels/lib/cpp/util/secret_reader.h>

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/io/node_table_reader.h>
#include <mapreduce/yt/interface/io.h>
#include <mapreduce/yt/io/job_reader.h>
#include <mapreduce/yt/io/job_writer.h>
#include <mapreduce/yt/io/node_table_writer.h>
#include <library/cpp/yson/node/node_io.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/protobuf/protofile/protofile.h>

#include <google/protobuf/descriptor.pb.h>

#include <google/protobuf/util/message_differencer.h>

#include <util/generic/maybe.h>
#include <util/generic/vector.h>
#include <util/generic/yexception.h>
#include <util/system/file.h>
#include <util/system/fs.h>

#include <util/datetime/base.h>
#include <util/generic/ptr.h>
#include <util/generic/string.h>
#include <util/system/event.h>
#include <util/stream/zlib.h>
#include <util/string/builder.h>
#include <util/thread/factory.h>

#include <functional>

namespace NTravel {

void InternalRegisterYtTableCache(const TString& name);

class TYtRetryConfigProviderFixed: public NYT::IRetryConfigProvider {
public:
    TYtRetryConfigProviderFixed(TDuration v)
        : RetriesTimeLimit(v)
    {}

    NYT::TRetryConfig CreateRetryConfig() override {
        NYT::TRetryConfig config;
        config.RetriesTimeLimit = RetriesTimeLimit;
        return config;
    }
private:
    TDuration RetriesTimeLimit;
};

template <class TProto>
class TYtTableCache: public IThreadFactory::IThreadAble {
public:
    using TNodeToProtoConverter = std::function<void (const NYT::TNode& node, TProto* proto)>;
    using TDataCallback = std::function<void (const TProto& proto)>;
    using TFinishCallback = std::function<void (bool ok, bool initial)>; // initial == true when first ok == true arrives

    TYtTableCache(const TString& name, const NTravelProto::NAppConfig::TYtTableCacheConfig& config)
        : LogPfx_(name + ": ")
        , Config_(config)
        , LoadFromFiles_(true)
    {
        InternalRegisterYtTableCache(name);
        if (!IsEnabled()) {
            return;
        }
        if (Config_.YtProxySize() == 0) {
            throw yexception() << "Cannot work without YtProxy: " << name;
        }
        if (Config_.YtTablePathSize() == 0) {
            throw yexception() << "Cannot work without YtTablePath: " << name;
        }
        NYT::TCreateClientOptions ytOpts;
        ytOpts.RetryConfigProvider(new TYtRetryConfigProviderFixed(TDuration::Seconds(Config_.GetYtTimeoutSec())));
        ytOpts.Token(ReadSecret(Config_.GetYtTokenPath()));

        for (const auto& ytProxy: Config_.GetYtProxy()) {
            YtClients_[ytProxy] = NYT::CreateClient(ytProxy, ytOpts);
        }
        TString cacheFilePath = Config_.GetCacheDir() + "/cache_" + name + ".pb";
        for (const auto& ytPath: Config_.GetYtTablePath()) {
            TTableInfo ti;
            ti.YtPath = ytPath;
            ti.LocalPath = cacheFilePath;
            if (Tables_){
                ti.LocalPath += "." + ToString(Tables_.size());
            }
            Tables_.push_back(ti);
        }
    }

    ~TYtTableCache() override {
        Stop();
    }

    bool IsEnabled() const {
        return Config_.GetEnabled();
    }

    void RegisterCounters(NMonitor::TCounterSource& source, const TString& name) {
        source.RegisterSource(&Counters_, name);
    }

    void SetCallbacks(TNodeToProtoConverter conv, TDataCallback data, TFinishCallback finish) {
        CbConv_ = conv;
        CbData_ = data;
        CbFinish_ = finish;
    }

    template <class TOwner>
    void SetCallbacks(TOwner* owner,
                      void (TOwner::*conv)(const NYT::TNode& node, TProto* proto) const,
                      void (TOwner::*data)(const TProto& proto),
                      void (TOwner::*finish)(bool ok, bool initial)) {
        SetCallbacks(
            [owner, conv](const NYT::TNode& node, TProto* proto) { (owner->*conv)(node, proto); },
            [owner, data](const TProto& proto) { (owner->*data)(proto); },
            [owner, finish](bool ok, bool initial) { (owner->*finish)(ok, initial); }
        );
    }

    void Start() {
        if (!IsEnabled()) {
            return;
        }
        if (StartFlag_.TrySet()) {
            Y_ASSERT(!Thread_);
            Thread_ = SystemThreadFactory()->Run(this);
        } else {
            WARNING_LOG << LogPfx_ << "Duplicate call of Start() is ignored" << Endl;
        }
    }

    void Stop() {
        if (!IsEnabled()) {
            return;
        }
        if (!StartFlag_) {
            WARNING_LOG << LogPfx_ << "Call of Stop() before Start()" << Endl;
            return;
        }
        if (StopFlag_.TrySet()) {
            WakeUp_.Signal();
            if (Thread_) {
                Thread_->Join();
                Thread_.Reset();
            }
        }
    }

    bool IsReady() const {
        return !Config_.GetEnabled() || DataLoaded_;
    }

    void Reload() {
        if (ReLoadFromFiles_.TrySet()) {
            WakeUp_.Signal();
        }
    }

private:
    enum class EError {
        Stopped,
        YtException,
        FileException,
        CallbackExcepiton,
        UnknownException,
    };

    struct TTypedException : public yexception {
        EError ErrorType;
        TTypedException(EError errorType)
            : ErrorType(errorType)
        {}
    };

    struct TCounters : public NMonitor::TCounterSource {
        NMonitor::TDerivCounter      NYtException;
        NMonitor::TDerivCounter      NFileException;
        NMonitor::TDerivCounter      NCallbackExcepiton;
        NMonitor::TDerivCounter      NUnknownException;
        NMonitor::TCounter           NTableCacheUpdateFailed; // постоянный счетчик
        NMonitor::TCounter           IsReady;
        NMonitor::TCounter           LoadingFromYt;
        NMonitor::TCounter           LoadingFromFile;

        void QueryCounters(NMonitor::TCounterTable* ct) const override {
            ct->insert(MAKE_COUNTER_PAIR(NYtException));
            ct->insert(MAKE_COUNTER_PAIR(NFileException));
            ct->insert(MAKE_COUNTER_PAIR(NCallbackExcepiton));
            ct->insert(MAKE_COUNTER_PAIR(NUnknownException));
            ct->insert(MAKE_COUNTER_PAIR(NTableCacheUpdateFailed));
            ct->insert(MAKE_COUNTER_PAIR(IsReady));
            ct->insert(MAKE_COUNTER_PAIR(LoadingFromYt));
            ct->insert(MAKE_COUNTER_PAIR(LoadingFromFile));
        }
    };

    struct TTableInfo {
        TString YtPath;
        TString LocalPath;
        TMaybe<TString> LocalModTime;
    };

    const TString LogPfx_;
    const NTravelProto::NAppConfig::TYtTableCacheConfig Config_;
    THashMap<TString/*YtProxy*/, NYT::IClientPtr> YtClients_;

    mutable TCounters Counters_;

    TNodeToProtoConverter CbConv_;
    TDataCallback CbData_;
    TFinishCallback CbFinish_;

    THolder<IThreadFactory::IThread> Thread_;
    TAtomicFlag StartFlag_;
    TAtomicFlag StopFlag_;
    TAutoEvent WakeUp_;

    TAtomicFlag LoadFromFiles_;
    TAtomicFlag ReLoadFromFiles_;
    TAtomicFlag DataLoaded_;
    TVector<TTableInfo> Tables_;

    void CheckSavedDescriptor(NFastTier::TBinaryProtoReader<>& reader, const ::google::protobuf::Descriptor* descriptor) const {
        ::google::protobuf::DescriptorProto actualDescriptorProto, fileDescriptorProto;
        descriptor->CopyTo(&actualDescriptorProto);
        if (!reader.GetNext(fileDescriptorProto)) {
            throw TTypedException(EError::FileException) << "No descriptor in file";
        }
        if (!::google::protobuf::util::MessageDifferencer::Equals(actualDescriptorProto, fileDescriptorProto)) {
            throw TTypedException(EError::FileException) << "Message descriptor in file differ from actual one";
        }
    }

    void WriteDescriptorToFile(NFastTier::TBinaryProtoWriter<>& writer, const ::google::protobuf::Descriptor* descriptor) const {
        ::google::protobuf::DescriptorProto actualDescriptorProto;
        descriptor->CopyTo(&actualDescriptorProto);
        writer.Write(actualDescriptorProto);
    }

    void WrappedCbData(const TProto& record) {
        try {
            CbData_(record);
        } catch (...) {
            throw TTypedException(EError::CallbackExcepiton) << "Exception during CbData: " << CurrentExceptionMessage();
        }
    }

    void WrappedCbConv(const NYT::TNode& node, TProto* proto) {
        try {
            CbConv_(node, proto);
        } catch (...) {
            throw TTypedException(EError::CallbackExcepiton) << "Exception during CbConv_: " << CurrentExceptionMessage();
        }
    }

    void WrappedCbFinish(bool success, bool initial) {
        try {
            CbFinish_(success, initial);
        } catch (...) {
            throw TTypedException(EError::CallbackExcepiton) << "Exception during CbFinish_: " << CurrentExceptionMessage();
        }
    }

    void LoadTableFromFile(bool withData, TTableInfo& ti) {
        TString excTxt = "Failed to load table from cache file " + ti.LocalPath;
        ti.LocalModTime.Clear();
        if (withData) {
            INFO_LOG << LogPfx_ << "Start loading table from cache file " << ti.LocalPath << Endl;
        } else {
            INFO_LOG << LogPfx_ << "Get modification time from cache file " << ti.LocalPath << Endl;
        }
        try {
            if (!NFs::Exists(ti.LocalPath)) {
                if (withData) {
                    throw TTypedException(EError::FileException) << "File not found";
                } else {
                    INFO_LOG << LogPfx_ << "Cache file " << ti.LocalPath << " does not exist" << Endl;
                    return;
                }
            }
            TFileInput file(ti.LocalPath);
            std::unique_ptr<TZLibDecompress> decompressStream;
            if (Config_.GetCompressed()) {
                decompressStream = std::make_unique<TZLibDecompress>(&file);
            }
            NFastTier::TBinaryProtoReader<> reader;
            reader.Open(decompressStream != nullptr ? (IInputStream*)decompressStream.get() : &file);
            NTravelProto::TTableCacheHeaderRecord header;
            CheckSavedDescriptor(reader, header.GetDescriptor());
            if (!reader.GetNext(header)) {
                throw TTypedException(EError::FileException) << "No header in file";
            }
            INFO_LOG << LogPfx_ << "Cache file " << ti.LocalPath << " has LocalModTime is " << header.GetModTime() << Endl;
            TProto record;
            CheckSavedDescriptor(reader, record.GetDescriptor());
            if (withData) {
                size_t rowIdx;
                for (rowIdx = 0; reader.GetNext(record); ++rowIdx) {
                    WrappedCbData(record);
                    if (StopFlag_) {
                        throw TTypedException(EError::Stopped) << "Abort due to stop";
                    }
                    if (rowIdx % 10000 == 0) {
                        INFO_LOG << LogPfx_ << rowIdx << " rows loaded from file " << ti.LocalPath << Endl;
                    }
                }
                INFO_LOG << LogPfx_ << rowIdx << " rows loaded from file " << ti.LocalPath << Endl;
                INFO_LOG << LogPfx_ << "Finished loading table from cache file " << ti.LocalPath << Endl;
            }
            ti.LocalModTime = header.GetModTime();
        } catch (const TTypedException& exc) {
            throw TTypedException(exc.ErrorType) << excTxt << ": " << exc.what();
        } catch (...) {
            throw TTypedException(EError::UnknownException) << excTxt << ": " << CurrentExceptionMessage();
        }
    }

    void LoadAllTablesFromFiles(bool withData) {
        NMonitor::TScopedCounterSetter counterSetter(Counters_.LoadingFromFile, 1, 0);
        try {
            for (auto& ti: Tables_) {
                LoadTableFromFile(withData, ti);
            }
            if (withData) {
                WrappedCbFinish(true, !DataLoaded_);
                DataLoaded_.Set();
                Counters_.IsReady = 1;
                LoadFromFiles_.Clear();
            }
        } catch (...) {
            if (withData) {
                WrappedCbFinish(false, false);
            }
            throw;
        }
    }

    void DownloadTableFromYtToFile(const TString& ytProxy, const TString& modTime, TTableInfo& ti) {
        NMonitor::TScopedCounterSetter counterSetter(Counters_.LoadingFromYt, 1, 0);
        TString excTxt = TStringBuilder() << "Failed to check table `" << ytProxy << "`." << ti.YtPath << " updates and save them to file " << ti.LocalPath;
        INFO_LOG << LogPfx_ << "Start loading from YT table `" << ytProxy << "`." << ti.YtPath << " to file " << ti.LocalPath << Endl;
        try {
            TString tempFilePath = ti.LocalPath + ".temp";
            auto outputFile = TFixedBufferFileOutput(tempFilePath);
            std::unique_ptr<TZLibCompress> compressStream;
            if (Config_.GetCompressed()) {
                compressStream = std::make_unique<TZLibCompress>(TZLibCompress::TParams(&outputFile));
            }
            NFastTier::TBinaryProtoWriter<> writer;
            try {
                writer.Open(compressStream != nullptr ? (IOutputStream*)compressStream.get() : &outputFile);
            } catch (...) {
                throw TTypedException(EError::CallbackExcepiton) << CurrentExceptionMessage();
            }
            TProto record;
            try {
                NTravelProto::TTableCacheHeaderRecord header;
                WriteDescriptorToFile(writer, header.GetDescriptor());
                header.SetModTime(modTime);
                writer.Write(header);
                WriteDescriptorToFile(writer, record.GetDescriptor());
            } catch (...) {
                throw TTypedException(EError::FileException) << CurrentExceptionMessage();
            }
            NYT::IClientPtr ytClient = YtClients_[ytProxy];
            auto reader = ytClient->CreateTableReader<NYT::TNode>(ti.YtPath);
            size_t rowIdx;
            for (rowIdx = 0; reader->IsValid(); reader->Next(), ++rowIdx) {
                record.Clear();
                WrappedCbConv(reader->GetRow(), &record);
                try {
                    writer.Write(record);
                } catch (...) {
                    throw TTypedException(EError::CallbackExcepiton) << CurrentExceptionMessage();
                }
                if (StopFlag_) {
                    throw TTypedException(EError::Stopped) << "Abort due to stop";
                }
                if (rowIdx % 10000 == 0) {
                    INFO_LOG << LogPfx_ << rowIdx << " rows loaded from YT to file " << tempFilePath << Endl;
                }
            }
            INFO_LOG << LogPfx_ << rowIdx << " rows loaded from YT to file " << tempFilePath << Endl;
            try {
                writer.Finish();
                if (!NFs::Rename(tempFilePath, ti.LocalPath)) {
                    throw yexception() << "Unable to rename '" << tempFilePath << "' to '" << ti.LocalPath << "', Error is '" << LastSystemErrorText() << "'";
                }
            } catch (...) {
                throw TTypedException(EError::FileException) << CurrentExceptionMessage();
            }
            INFO_LOG << LogPfx_ << "Finished loading from YT table to file " << ti.LocalPath << ", new modTime is " << modTime << Endl;
            ti.LocalModTime = modTime;
        } catch (const TTypedException& exc) {
            throw TTypedException(exc.ErrorType) << excTxt << ": " << exc.what();
        } catch (...) {
            throw TTypedException(EError::YtException) << excTxt << ": " << CurrentExceptionMessage();
        }
    }

    void GetMaxYtModTime(TTableInfo& ti, TString* maxYtModTime, TString* maxYtProxy) {
        maxYtModTime->clear();
        for (const auto& [ytProxy, ytClient]: YtClients_) {
            try {
                DEBUG_LOG << LogPfx_ << "Getting modtime of table `" << ytProxy << "`." << ti.YtPath << Endl;
                TString modTime = ytClient->Get(ti.YtPath + "/@modification_time").AsString();
                DEBUG_LOG << LogPfx_ << "Got modtime of table `" << ytProxy << "`." << ti.YtPath << ": " << modTime << Endl;
                if (modTime > *maxYtModTime) {
                    *maxYtModTime = modTime;
                    *maxYtProxy = ytProxy;
                }
            } catch (...) {
                ERROR_LOG << LogPfx_ << "Failed to get modtime of table `" << ytProxy << "`." << ti.YtPath << ": " << CurrentExceptionMessage() << Endl;
            }
        }
        if (maxYtModTime->empty()) {
            throw TTypedException(EError::YtException) << "YT not acessible";
        }
    }

    void DownloadAllChages()  {
        for (TTableInfo& ti: Tables_) {
            TString ytProxy;
            TString maxYtModTime;
            GetMaxYtModTime(ti, &maxYtModTime, &ytProxy);
            if (ti.LocalModTime != maxYtModTime) {
                DownloadTableFromYtToFile(ytProxy, maxYtModTime, ti);
                LoadFromFiles_.Set();// Data should be reloaded from file
            }
        }
    }

    void HandleExceptions(std::function<void()> func, bool* ytError = nullptr, bool* otherErrors = nullptr) {
        TMaybe<EError> error;
        try {
            func();
        } catch (const TTypedException& exc) {
            ERROR_LOG << LogPfx_ << exc.what() << Endl;
            error = exc.ErrorType;
        } catch (...) {
            ERROR_LOG << LogPfx_ << "Unknown exception: " << CurrentExceptionMessage() << Endl;
            error = EError::UnknownException;
        }
        if (error) {
            if (error.GetRef() == EError::YtException) {
                if (ytError) {
                    *ytError = true;
                }
            } else {
                if (otherErrors) {
                    *otherErrors = true;
                }
            }
            switch (error.GetRef()) {
                case EError::Stopped:
                    break;
                case EError::YtException:
                    Counters_.NYtException.Inc();
                    break;
                case EError::FileException:
                    Counters_.NFileException.Inc();
                    break;
                case EError::CallbackExcepiton:
                    Counters_.NCallbackExcepiton.Inc();
                    break;
                case EError::UnknownException:
                    Counters_.NUnknownException.Inc();
                    break;
            }
        }
    }

    void DoExecute() override {
        HandleExceptions([this](){ LoadAllTablesFromFiles(false); });
        while (!StopFlag_) {
            bool ytError = false;
            bool otherErrors = false;
            if (ReLoadFromFiles_) {
                ReLoadFromFiles_.Clear();
                LoadFromFiles_.Set();
                INFO_LOG << LogPfx_ << "Reloading from file forced" << Endl;
            } else {
                HandleExceptions([this](){ DownloadAllChages(); }, &ytError, &otherErrors);
            }
            if (LoadFromFiles_) {
                HandleExceptions([this](){ LoadAllTablesFromFiles(true); }, &ytError, &otherErrors);
            }
            if (StopFlag_) {
                return;
            }
            // Алерт только если нет данных, или же любая ошибка кроме YT
            Counters_.NTableCacheUpdateFailed = (!DataLoaded_ || otherErrors) ? 1 : 0;
            if (ReLoadFromFiles_) {
                // Не ждём, перечитываем сразу
                continue;
            }
            // При любой ошибке - пробуем чаще
            int pollPeriod = (ytError || otherErrors) ? Config_.GetPollPeriodOnErrorSec() : Config_.GetPollPeriodSec();
            WakeUp_.WaitT(TDuration::Seconds(pollPeriod));
        }
    }
};

} // namespace NTravel
