package st

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"

	"github.com/go-resty/resty/v2"
	"github.com/mitchellh/mapstructure"
)

// StartrekClient provides an interface for Tracker.
type StartrekClient struct {
	URL   string
	Token string
	OrgID string
}

// NewStartrekClientForProduction returns a client instance pointing to the production environment.
func NewStartrekClientForProduction(token string) *StartrekClient {
	return &StartrekClient{
		URL:   "api.tracker.yandex.net",
		Token: token,
	}
}

// NewStartrekClientForOrg returns a client instance pointing to the B2B tracker of a given organization
func NewStartrekClientForOrg(token string, ordID string) *StartrekClient {
	return &StartrekClient{
		URL:   "api.tracker.yandex.net",
		Token: token,
		OrgID: ordID,
	}
}

func (c *StartrekClient) baseURL() string {
	return "https://" + c.URL + "/v2"
}

func (c *StartrekClient) r(ctx context.Context) *resty.Request {
	req := resty.New().R().
		SetHeader("Authorization", "OAuth "+c.Token).
		SetContext(ctx).
		ExpectContentType("application/json")
	if c.OrgID != "" {
		req.SetHeader("X-Org-Id", c.OrgID)
	}
	return req
}

////////////////////////////////////////////////////////////////////////////////
// Models.

type Ref struct {
	Self    string `mapstructure:"self,omitempty"`
	ID      string `mapstructure:"id,omitempty"`
	Key     string `mapstructure:"key,omitempty"`
	Display string `mapstructure:"display,omitempty"`
}

type LinkType struct {
	Ref Ref `mapstructure:",squash"`

	Inward  string `mapstructure:"inward,omitempty"`
	Outward string `mapstructure:"outward,omitempty"`
}

type Link struct {
	Ref Ref `mapstructure:",squash"`

	Type      LinkType `mapstructure:"type"`
	Direction string   `mapstructure:"direction,omitempty"`
	Object    Ref      `mapstructure:"object,omitempty"`
}

type Issue struct {
	Ref Ref `mapstructure:",squash"`

	Version int64 `mapstructure:"version,omitempty"`

	Summary     string   `mapstructure:"summary,omitempty"`
	Description string   `mapstructure:"description,omitempty"`
	Tags        []string `mapstructure:"tags,omitempty"`

	Favorite bool `mapstructure:"favorite,omitempty"`

	Queue      *Ref `mapstructure:"queue,omitempty"`
	Status     *Ref `mapstructure:"status,omitempty"`
	Resolution *Ref `mapstructure:"resolution,omitempty"`
	Type       *Ref `mapstructure:"type,omitempty"`
	Priority   *Ref `mapstructure:"priority,omitempty"`

	Assignee  *Ref   `mapstructure:"assignee,omitempty"`
	Followers []*Ref `mapstructure:"followers,omitempty"`

	CreatedBy  *Ref `mapstructure:"createdBy,omitempty"`
	UpdatedBy  *Ref `mapstructure:"updatedBy,omitempty"`
	ResolvedBy *Ref `mapstructure:"resolvedBy,omitempty"`

	CreatedAt  string `mapstructure:"createdAt,omitempty"`
	UpdatedAt  string `mapstructure:"updatedAt,omitempty"`
	ResolvedAt string `mapstructure:"resolvedAt,omitempty"`

	Links       []*Link `mapstructure:"links,omitempty"`
	RemoteLinks []*Ref  `mapstructure:"remotelinks,omitempty"`

	Epic   *Ref `mapstructure:"epic,omitempty"`
	Parent *Ref `mapstructure:"parent,omitempty"`

	Other map[interface{}]interface{} `mapstructure:",remain"`
}

////////////////////////////////////////////////////////////////////////////////

// https://st-api.yandex-team.ru/docs/#operation/getIssueApiV2
type GetIssueRequest struct {
	// Path parameter.
	ID string

	// Query parameters.
	Fields  []string
	Expand  []string
	Embed   []string
	StaleOk bool
}

type GetIssueResponse struct {
	Issue *Issue
}

func (r *GetIssueResponse) UnmarshalJSON(data []byte) error {
	return unmarshalJSONWithMapstructure(data, &r.Issue)
}

// https://st-api.yandex-team.ru/docs/#operation/createIssueApiV2
type CreateIssueRequest struct {
	// Query parameters.
	Notify       bool
	NotifyAuthor bool
	Recipients   []string
	Fields       []string
	Expand       []string
	Embed        []string
	StaleOk      bool

	// Body parameters.
	Body struct {
		Queue       string   `json:"queue"`
		Summary     string   `json:"summary"`
		Description string   `json:"description"`
		Tags        []string `json:"tags,omitempty"`
	}
}

type CreateIssueResponse struct {
	Issue *Issue
}

func (r *CreateIssueResponse) UnmarshalJSON(data []byte) error {
	return unmarshalJSONWithMapstructure(data, &r.Issue)
}

func unmarshalJSONWithMapstructure(input []byte, output interface{}) error {
	var dynamic map[string]interface{}
	if err := json.Unmarshal(input, &dynamic); err != nil {
		return err
	}
	if err := mapstructure.Decode(dynamic, output); err != nil {
		return err
	}
	return nil
}

func addStringListParam(params url.Values, key string, values []string) {
	switch len(values) {
	case 0:
		params.Del(key)
	default:
		for _, value := range values {
			params.Add(key, value)
		}
	}
}

func (req *GetIssueRequest) Execute(client *StartrekClient, ctx context.Context) (*GetIssueResponse, error) {
	params := make(url.Values)
	addStringListParam(params, "fields", req.Fields)
	addStringListParam(params, "expand", req.Expand)
	addStringListParam(params, "embed", req.Embed)
	if req.StaleOk {
		params.Add("staleOk", "true")
	}
	var result GetIssueResponse
	httpreq := client.r(ctx).
		SetResult(&result).
		SetPathParams(map[string]string{"id": req.ID}).
		SetQueryParamsFromValues(params)
	if httprsp, err := httpreq.Get(client.baseURL() + "/issues/{id}/"); err != nil {
		return nil, err
	} else if httprsp.StatusCode() != http.StatusOK {
		return nil, fmt.Errorf("bad status code: %v", httprsp.Status())
	} else {
		return &result, nil
	}
}

func (req *CreateIssueRequest) Execute(client *StartrekClient, ctx context.Context) (*CreateIssueResponse, error) {
	params := make(url.Values)
	params.Add("notify", fmt.Sprint(req.Notify))
	params.Add("notifyAuthor", fmt.Sprint(req.NotifyAuthor))
	addStringListParam(params, "recipients", req.Recipients)
	addStringListParam(params, "fields", req.Fields)
	addStringListParam(params, "expand", req.Expand)
	addStringListParam(params, "embed", req.Embed)
	if req.StaleOk {
		params.Add("staleOk", "true")
	}
	var result CreateIssueResponse
	httpreq := client.r(ctx).
		SetResult(&result).
		SetQueryParamsFromValues(params).
		SetBody(&req.Body)
	if httprsp, err := httpreq.Post(client.baseURL() + "/issues/"); err != nil {
		return nil, err
	} else if httprsp.StatusCode() != http.StatusCreated {
		return nil, fmt.Errorf("bad status code: %v", httprsp.Status())
	} else {
		return &result, nil
	}
}
