package ytqueue

import (
	"context"
	"fmt"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
)

type BusRow struct {
	TabletIDX   uint64 `yson:"$tablet_index"`
	RowIDX      uint64 `yson:"$row_index"`
	MessageID   string `yson:"MessageId"`
	Timestamp   uint64 `yson:"Timestamp"`
	MessageType string `yson:"MessageType"`
	Codec       int    `yson:"Codec"`
	Bytes       []byte `yson:"Bytes"`
}

type readResult struct {
	cluster string
	result  *BusRow
	err     error
}

type QueueConfig struct {
	Proxy    string
	Path     string
	Token    string
	Revision uint64
}

const readLimit = 3000
const dedupWindow = time.Minute * 5

func merge(ctx context.Context, chans []<-chan readResult) <-chan readResult {
	var wg sync.WaitGroup
	out := make(chan readResult)
	output := func(c <-chan readResult) {
		defer wg.Done()
		for n := range c {
			select {
			case out <- n:
				continue
			case <-ctx.Done():
				return
			}
		}
	}
	wg.Add(len(chans))
	for _, c := range chans {
		go output(c)
	}
	go func() {
		wg.Wait()
		close(out)
	}()
	return out
}

func Read(ctx context.Context, clusters []QueueConfig, logger log.Logger) <-chan *BusRow {
	result := make(chan *BusRow)
	chans := make([]<-chan readResult, len(clusters))
	for i, cluster := range clusters {
		client, err := ythttp.NewClient(&yt.Config{
			Proxy: cluster.Proxy,
			Token: cluster.Token,
		})
		if err != nil {
			panic(err)
		}
		chans[i] = readCluster(ctx, client, cluster.Path, cluster.Revision, readLimit)
	}

	go func() {
		nested, cancel := context.WithCancel(ctx)
		defer func() {
			cancel()
			close(result)
		}()
		dedup := newDeduplicationCache(dedupWindow, logger)
		merged := merge(nested, chans)
		for rr := range merged {
			if rr.err != nil {
				fmt.Println(rr.err)
				continue
			} else {
				if dedup.isKnown(rr.result.MessageID) {
					continue
				} else {
					select {
					case result <- rr.result:
						continue
					case <-ctx.Done():
						return
					}
				}
			}
		}
	}()
	return result
}

func readCluster(ctx context.Context, ytc yt.Client, path string, revision uint64, limit uint64) <-chan readResult {
	resultChannel := make(chan readResult)
	go func() {
		defer close(resultChannel)
		for {
			var cont bool
			cont, revision = readClusterAttempt(ctx, ytc, path, revision, limit, resultChannel)
			if !cont {
				return
			}
		}
	}()
	return resultChannel
}

func readClusterAttempt(ctx context.Context, ytc yt.Client, path string, revision uint64, limit uint64, resultChannel chan readResult) (bool, uint64) {
	var query string
	if revision != 0 {
		query = fmt.Sprintf("* FROM [%s] WHERE [$tablet_index] = 0 AND [$row_index] > %d AND [$row_index] <= %d", path, revision, revision+limit)
	} else {
		query = fmt.Sprintf("* FROM [%s] LIMIT %d", path, limit)
	}
	rows, err := ytc.SelectRows(ctx, query, nil)
	if err != nil {
		result := readResult{
			err: err,
		}
		select {
		case resultChannel <- result:
			time.Sleep(time.Second)
			return true, revision
		case <-ctx.Done():
			return false, revision
		}
	}
	defer rows.Close()
	dataWasRead := false
	for rows.Next() {
		var row BusRow
		result := readResult{}
		err = rows.Scan(&row)
		if err != nil {
			result.err = err
		} else {
			dataWasRead = true
			result.result = &row
			revision = row.RowIDX
		}
		select {
		case resultChannel <- result:
			continue
		case <-ctx.Done():
			return false, revision
		}
	}

	if err := rows.Err(); err != nil {
		result := readResult{
			err: err,
		}
		select {
		case resultChannel <- result:
			time.Sleep(time.Second)
			return true, revision
		case <-ctx.Done():
			return false, revision
		}
	}

	if !dataWasRead {
		time.Sleep(time.Second)
	}
	return true, revision
}
